from __future__ import absolute_import, print_function, division

import argparse
import msgpack
import os
import pprint
import random
import socket
import subprocess
import time

from library.config import query

from ..cmscache import getHostList


def parse_args():
    parser = argparse.ArgumentParser()

    parser.add_argument(
        '-f', '--format', choices=('pretty', 'msgpack'), default='pretty',
        help='Output format'
    )
    parser.add_argument(
        '--rundir', default=None, type=str, help='heartbeat-client rundir',
    )

    return parser.parse_args()


def get_fastbone_fqdn(fqdn):
    if fqdn.endswith('.search.yandex.net'):
        return 'fb-' + fqdn
    if fqdn.endswith('.yandex.ru'):
        return fqdn.replace('.yandex.ru', '.fb.yandex.ru')
    return fqdn


def normalize(target, net):
    fqdn = socket.getfqdn(target)

    if net == 'bb4':
        return socket.AF_INET, 'ping', fqdn
    if net == 'bb6':
        return socket.AF_INET6, 'ping6', fqdn.replace('.yandex.ru', '.search.yandex.net')
    if net == 'fb6':
        return socket.AF_INET6, 'ping6', get_fastbone_fqdn(fqdn)


def am_i_ipv6_only():
    if os.uname()[0] == 'FreeBSD':
        return False

    # noinspection PyBroadException
    try:
        out, err = subprocess.Popen(
            ["ip -4 route | wc -l"],
            shell=True,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE
        ).communicate()

        return int(out) == 0
    except:
        return False


def check(target, net, proto='icmp', port=None):
    for i in range(0, 3):
        # noinspection PyBroadException
        try:
            family, ping_cmd, fqdn = normalize(target, net)
            ip = socket.getaddrinfo(fqdn, 0, family, socket.SOCK_STREAM, socket.SOL_TCP)[0][4][0]
        except socket.error:
            if proto == 'tcp':
                return {'result': -1.0, 'connect_time_sec': 0.0}
            return -1.0

        if proto == 'icmp':
            # noinspection PyBroadException
            try:
                no_fragmentation_key = '-D' if os.uname()[0] == 'FreeBSD' else '-M do'

                out, err = subprocess.Popen(
                    ["{0} -q -c 10 -s 8000 {1} {2} ".format(ping_cmd, no_fragmentation_key, ip) +
                     "| grep transmitted | awk '{print $4}'"],
                    shell=True,
                    stdout=subprocess.PIPE,
                    stderr=subprocess.PIPE
                ).communicate()

                return float(out)/10
            except:
                continue

        elif proto == 'udp':
            msg = 'HELLO'
            family, _, source_fqdn = normalize(socket.getfqdn(), net)
            s = socket.socket(family, socket.SOCK_DGRAM)
            s.bind((source_fqdn, 0))
            try:
                s.sendto(msg, (fqdn, port))
                s.settimeout(5.0)
                res, _ = s.recvfrom(1024)
                s.close()
                return 1.0 if res == msg else 0.0
            except socket.error:
                continue

        elif proto == 'tcp':
            s = socket.socket(family, socket.SOCK_STREAM)
            s.settimeout(10.0)
            try:
                t = time.time()
                s.connect((fqdn, port))
                dt = time.time() - t
                s.close()
                return {'result': 1.0, 'connect_time_sec': dt}

            except socket.error:
                return {'result': 0.0, 'connect_time_sec': 0}

            except:
                return {'result': -1.0, 'connect_time_sec': 0}

        elif proto == 'netliba':
            pass

        else:
            raise IndexError

    return 0.0


def main():
    args = parse_args()
    if args.rundir is None:
        from api.srvmngr import getRoot
        args.rundir = os.path.join(getRoot(), 'var', 'heartbeat-client')

    target = random.choice(getHostList('ALL_SEARCH', 7200, args.rundir))
    conf = query('skynet.services.netstate', 'config')

    bb6_tcp = check(target, 'bb6', 'tcp', conf['port'])
    fb6_tcp = check(target, 'fb6', 'tcp', conf['port'])

    result = {
        'source': socket.getfqdn(),
        'target': socket.getfqdn(target),
        'bb6': check(target, 'bb6'),
        'bb6_tcp': bb6_tcp['result'],
        'bb6_connect_time_sec': bb6_tcp['connect_time_sec'],
        'bb6_udp': check(target, 'bb6', 'udp', conf['port']),
        'fb6': check(target, 'fb6'),
        'fb6_tcp': fb6_tcp['result'],
        'fb6_tcp_connect_time_sec': fb6_tcp['connect_time_sec'],
        'fb6_udp': check(target, 'fb6', 'udp', conf['port']),
    }

    if not am_i_ipv6_only():
        bb4_tcp = check(target, 'bb4', 'tcp', conf['port'])
        result['bb4'] = check(target, 'bb4')
        result['bb4_tcp'] = bb4_tcp['result']
        result['bb4_tcp_connect_time_sec'] = bb4_tcp['connect_time_sec']
        result['bb4_udp'] = check(target, 'bb4', 'udp', conf['port'])

    if args.format == 'pretty':
        pprint.pprint(result)
    if args.format == 'msgpack':
        print(msgpack.packb(result))


if __name__ == '__main__':
    main()
