from __future__ import absolute_import, print_function, division

import argparse
import msgpack
import py
import re
import math

from pprint import pprint


REPORT_VERSION = 1
FORMATS = ('msgpack', 'pp')
DB_BASE_PATH = py.path.local('/db/BASE')

okconv = lambda s: True if s == 'OK' else s


def shard_state():
    shards = {}
    result = {
        'base': DB_BASE_PATH.strpath,
        'version': REPORT_VERSION,
        'shards': shards,
    }

    try:
        if not DB_BASE_PATH.check(exists=1):
            result['error'] = 'Base path not exists'
        elif not DB_BASE_PATH.check(dir=1):
            result['error'] = 'Base path is not a directory'
        else:
            seen = set()
            for path in DB_BASE_PATH.listdir():
                rpath = path.realpath()
                if rpath in seen:
                    continue
                seen.add(rpath)

                name = rpath.basename

                shardstate = {}
                shards[name] = shardstate

                try:
                    statefile = rpath.join('shard.state')
                    if not statefile.check(exists=1):
                        shardstate['error'] = 'shard.state file does not exists'
                    else:
                        for line in statefile.readlines():
                            line = line.strip()
                            if not line:
                                continue
                            if ':' in line:
                                key, value = line.split(':', 1)
                                key, value = key.strip(), value.strip()

                                if value == 'undef':
                                    continue

                                shardstate.setdefault('state', {})[key] = value
                            else:
                                shardstate.setdefault('warnings', []).append('Cant parse line in shard.state: %r' % (line, ))

                    conffile = rpath.join('shard.conf')

                    shardstate.setdefault('state', {})
                    shardstate['state']['total-size'] = 0
                    shardstate['state']['acquired-size'] = 0

                    if conffile.check(exists=1):

                        reading_attrs = False
                        files = {}

                        attr_reg = re.compile('^%attr\((?P<attrs>.*)\)\s+(?P<filename>.*)$', re.I)
                        conf_mtime = 0
                        for line in conffile.readlines():
                            line = line.strip()
                            if not line:
                                continue

                            if line.startswith('%mtime'):
                                conf_mtime = int(line.split(' ')[1])
                                continue

                            if reading_attrs:
                                if line.startswith('%attr'):
                                    match = attr_reg.match(line)
                                    if match:
                                        match = match.groupdict()
                                        attrs = dict(
                                            [(s.strip().split('=', 1) if '=' in s else (s.strip(), None))  for s in match['attrs'].split(',')]
                                        )

                                        filename = match.get('filename', None)
                                        if filename:
                                            if 'size' in attrs and attrs.get('size', None).isdigit():
                                                size = attrs['size']
                                                files[filename] = int(size)
                                    else:
                                        shardstate.setdefault('warnings', []).append('Cant parse line in shard.conf: %r' % (line, ))
                                else:
                                    break
                            else:
                                if line.startswith('%files'):
                                    reading_attrs = True

                        shardstate['state']['conf_mtime'] = conf_mtime

                        total_size = 0
                        acquired_size = 0

                        for fn, size in files.items():
                            if size == 0:
                                continue
                            total_size += size
                            file_ = rpath.join(fn)
                            if file_.check(exists=1):
                                stat = file_.stat()
                                if stat.blocks * 512 > size:
                                    acquired_size += size
                                else:
                                    acquired_size += stat.blocks * 512

                        shardstate['state']['total-size'] = total_size
                        shardstate['state']['acquired-size'] = acquired_size

                        indexinv_file = rpath.join('indexinv')
                        try:
                            shardstate['state']['indexinv-size'] = indexinv_file.size()
                        except:
                            shardstate['state']['indexinv-size'] = None

                except Exception as ex:
                    shards[name] = {'error': str(ex)}
    except Exception as ex:
        result['error'] = str(ex)

    return result


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-f', '--format', choices=('pretty', 'msgpack'), default='pretty')
    return parser.parse_args()


def main():
    args = parse_args()

    result = {
        'report_version': REPORT_VERSION,
    }

    result = shard_state()

    if args.format == 'pretty':
        pprint(result)
    if args.format == 'msgpack':
        print(msgpack.packb(result))


if __name__ == '__main__':
    main()
