from __future__ import absolute_import, print_function, division
import argparse
import msgpack
import py
import re
from pprint import pprint

REPORT_VERSION = 3

BSCONFIG_PATH = py.path.local('/db/BASE')
ISS_PATH = py.path.local('/db/iss3/shards')

STATE_FILE = 'shard.state'
CONF_FILE = 'shard.conf'
TAG_FILE = 'stamp.TAG'
SPAM_DIR = 'Spam'
SPAM_FILE = 'bans.rev'


# a  - 1/0 - shard acquired yes/no
# as - indexarc file size
# cs - int hash from generated_chksum(bsconfig_shards.state)
# ct - %mtime(bsconfig_shard.conf)
# ds - sum of downloaded(acquired) sizes of shard files
# i  - 1/0 - installed yes/no
# is - indexinv file size
# s  - shards
# sv - Spam bans.rev
# ts - sum of sizes of shard files
# v  - report version
# z  - stamp.TAG SearchZone

def shardstate():
    shards = {}
    result = {'v': REPORT_VERSION, 's': shards}

    for shard_dir in [BSCONFIG_PATH, ISS_PATH]:
        try:
            if not shard_dir.check(exists=1) or not shard_dir.check(dir=1):
                continue
        except py.error.Error:
            continue

        seen_rpath = set()
        for path in shard_dir.listdir():
            rpath = path.realpath()
            if rpath.basename == 'lost+found':
                continue
            if rpath in seen_rpath:
                continue
            seen_rpath.add(rpath)
            shards.setdefault(rpath.basename, {})
            shards[rpath.basename].update(shardstate_by_rpath(rpath))

    return result


def shardstate_by_rpath(rpath):
    state = {}
    for f in [state_file_parse, conf_file_parse, tag_file_parse, spam_parse]:
        state.update(f(rpath))
    return state


def state_file_parse(rpath):
    def int_ok(x):
        return 1 if x == 'OK' else 0

    state = {'a': 0, 'i': 0, 'cs': 0}

    try:
        state_file = rpath.join(STATE_FILE)
        for line in state_file.readlines():
            line = line.strip()
            if line and ':' in line:
                key, value = line.split(':', 1)
                key, value = key.strip(), value.strip()
                if key == 'acquire':
                    state['a'] = int_ok(value)
                elif key == 'install':
                    state['i'] = int_ok(value)
                elif key == 'generated_chksum':
                    state['cs'] = value.__hash__()
    except py.error.Error:
        pass

    if state['i'] == 1:
        state['a'] = 1

    return state


def conf_file_parse(rpath):
    attr_reg = re.compile('^%attr\((?P<attrs>.*)\)\s+(?P<filename>.*)$', re.I)
    state = {'ts': 0, 'ds': 0, 'ct': 0, 'is': 0, 'as': 0}
    try:
        files = {}
        conf_file = rpath.join(CONF_FILE)
        for line in conf_file.readlines():
            line = line.strip()
            if not line:
                continue
            if line.startswith('%mtime'):
                state['ct'] = int(line.split(' ')[1])
            if line.startswith('%attr'):
                match = attr_reg.match(line)
                if match:
                    match = match.groupdict()
                    attrs = dict(
                        [(s.strip().split('=', 1) if '=' in s else (s.strip(), None))
                         for s in match['attrs'].split(',')]
                    )
                    filename = match.get('filename', None)
                    if filename:
                        if 'size' in attrs and attrs.get('size', None).isdigit():
                            size = attrs['size']
                            files[filename] = int(size)
        for fn, size in files.items():
            state['ts'] += size
            file_ = rpath.join(fn)
            if file_.check(exists=1):
                stat = file_.stat()
                if stat.blocks * 512 > size:
                    state['ds'] += size
                else:
                    state['ds'] += stat.blocks * 512
        indexinv_file = rpath.join('indexinv')
        if indexinv_file.check(exists=1):
            state['is'] = indexinv_file.size()
        indexarc_file = rpath.join('indexarc')
        if indexarc_file.check(exists=1):
            state['as'] = indexarc_file.size()
    except py.error.Error:
        pass
    return state


def tag_file_parse(rpath):
    state = {'z': ''}
    try:
        tag_file = rpath.join(TAG_FILE)
        for line in tag_file.readlines():
            line = line.strip()
            if line and '=' in line:
                key, value = line.split('=', 1)
                key, value = key.strip(), value.strip()
                if key == 'SearchZone':
                    state['z'] = value
    except py.error.Error:
        pass
    return state


def spam_parse(rpath):
    state = {'sv': -1}
    try:
        state_file = rpath.join(SPAM_DIR).join(SPAM_FILE)
        for line in state_file.readlines():
            line = line.strip()
            if line:
                state['sv'] = int(line)
    except py.error.Error:
        pass
    return state


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-f', '--format', choices=('pretty', 'msgpack'), default='pretty')
    return parser.parse_args()


def main():
    args = parse_args()
    result = shardstate()
    if args.format == 'pretty':
        pprint(result)
    if args.format == 'msgpack':
        print(msgpack.packb(result))


if __name__ == '__main__':
    main()
