#!/skynet/python/bin/python

from __future__ import absolute_import, print_function, division

import argparse
import msgpack
import subprocess
import time
import psutil

from py.path import local as Path  # noqa
from library.tasks import version
from api.srvmngr import getRoot


def get_deploy_type():
    fp = Path('/Berkanavt/skynet/etc/release.conf')
    if fp.check(exists=1) and fp.check(file=1):
        return 'meta'

    fp = Path('/Berkanavt/skynet/bin/gosky')
    if fp.check(exists=1) and fp.check(file=1):
        fp = Path('/var/run/gosky.nocron')
        return 'gosky_no_cron' if fp.check(exists=1) and fp.check(file=1) else 'gosky'

    return 'no_gosky'


def get_memory_usage(paths):
    values = ['rss', 'mapped_file']
    result = {}
    for path in paths:
        try:
            with open('/sys/fs/cgroup/memory/{}/memory.stat'.format(path), 'r') as f:
                size = 0
                for line in f:
                    chunks = line.split()
                    if len(chunks) > 1 and chunks[0] in values:
                        size += int(chunks[1]) / (1024 * 1024.0)

                result[path] = size

        except BaseException as ex:
            pass

    return result


def _get_porto_version_using_api():
    result = {}
    try:
        import porto
        conn = porto.Connection()
        conn.TryConnect()

        # are we on host or in container?
        abs_name = conn.GetProperty('self', 'absolute_name')
        if abs_name == '/':
            version = conn.Version()
        else:
            version = ('in_container', '')

        result['client'] = version[0]
        result['server'] = version[0]
    except AttributeError:
        pass
    except BaseException:
        # something goes wrong -> handle as no porto
        result = {'client': 'no', 'server': 'no'}

    return result


def _get_porto_version_using_portoctl():
    result = {}
    try:
        # API is old, try portoctl
        proc = subprocess.Popen(
            ['/usr/sbin/portoctl', '--version'],
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE
        )

        timeout = 10
        step = 0.1
        while timeout:
            time.sleep(step)
            if proc.poll() is not None:
                out, err = proc.communicate()
                # result can be in out or err
                for s in [out, err]:
                    for line in s.split('\n'):
                        r = line.strip().split(': ')
                        if len(r) == 2:
                            result[r[0]] = r[1]
                break
            timeout -= step

        if not timeout:
            # portoctl does not respond, kill it!
            result = {'client': 'timeout', 'server': 'timeout'}
            proc.kill()
    except BaseException:
        pass

    return result


def get_porto_versions():
    return _get_porto_version_using_api() or _get_porto_version_using_portoctl() or {'client': 'no', 'server': 'no'}


def get_disk_usage():
    d = psutil.disk_usage(getRoot())
    return d.percent


def find_cron_process():
    for p in psutil.process_iter():
        if p.name == 'cron':
            return True

    return False


def get_config_overrides():
    fp = Path(getRoot() + '/skycore/conf/overrides.d/')
    return fp.check(exists=1) and fp.check(dir=1) and len(fp.listdir()) > 0


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-f', '--format', choices=('pretty', 'msgpack'), default='pretty')
    parser.add_argument('--cgroup-check-path', action='append', default=[])
    return parser.parse_args()


def main():
    args = parse_args()

    info = version.VersionRunner().run()
    info['srvmngr'] = {
        'memory': get_memory_usage(args.cgroup_check_path),
        'deploy_type': get_deploy_type(),
        'disk_usage': get_disk_usage(),
        'cron_running': find_cron_process(),
        'overrides': get_config_overrides(),
    }
    info['porto'] = get_porto_versions()

    if args.format == 'msgpack':
        print(msgpack.dumps(info))
    elif args.format == 'pretty':
        from pprint import pprint as pp
        pp(info)


if __name__ == '__main__':
    main()
