#!/skynet/python/bin/python

from __future__ import absolute_import, print_function

import re
import argparse
import os
import socket
import multiprocessing
import subprocess
import json
import time
import requests
import signal
import fcntl
import logging

import porto
from kernel.util.sys.gettime import monoTime

try:
    if os.uname()[0] == 'Linux':
        import porto
except:
    pass

import msgpack

from .common import ISS_ILIST_URL, EInstanceTypes, _update_iss_instance_env, instance_major_tag, instance_minor_tag, instance_version, instance_group
from .openstack_utils import get_openstack_instances, get_tenant_id


# =====================================================================================
# Auxiliarily functions
# =====================================================================================

def _host_ncpu():
    if os.uname()[0] == 'FreeBSD':
        retcode, out, err = run_command(["sysctl", "hw.ncpu"])
        return int(out.strip().split(' ')[1])
    elif os.uname()[0] == 'Linux':
        return int(os.sysconf('SC_NPROCESSORS_ONLN'))


def _detect_child_pids_recurse(pid):
    try:
        code, out, err = run_command(["pgrep", "-P", str(pid)])
    except Exception:
        return []

    result = []
    child_pids = map(lambda x: int(x), out.split())
    for child_pid in child_pids:
        result.extend(_detect_child_pids_recurse(child_pid))
        result.append(child_pid)

    return result


def _detect_pids(instance_env):
    pids = []

    if instance_env["type"] == EInstanceTypes.BSCONFIG:
        piddir = os.path.join(instance_env["env"]["dir"], "pids")
        if os.path.exists(piddir):
            for fname in os.listdir(piddir):
                try:
                    with file(os.path.join(piddir, fname)) as f:
                        pids.append(int(f.read()))
                except Exception:  # some files are empty, so just skip them
                    pass

        if ('a_itype_upper' in instance_env["env"]["tags"]) or ('a_itype_newsupper' in instance_env["env"]["tags"]):
            upperapache_pid_file = os.path.join(instance_env["env"]["index"]["dir"], "arkanavt", "upper_bundle",
                                                "httpd.pid")
            try:
                with file(upperapache_pid_file) as f:
                    pids.append(int(f.read()))
            except Exception:
                pass

        if 'a_itype_vm' in instance_env["env"]["tags"]:
            try:
                retcode, out, err = run_command(["pidof", "kvm"])
                pids.append(int(out))
            except Exception:
                pass

    elif instance_env["type"] == EInstanceTypes.OPENSTACK:
        pids = [instance_env["env"]["pid"]]

    elif instance_env["type"] in (EInstanceTypes.GENCFG, EInstanceTypes.QLOUD, EInstanceTypes.YP):
        pass

    pids = filter(lambda pid: os.path.exists("/proc/%s" % pid), pids)
    pids = pids + sum(map(lambda x: _detect_child_pids_recurse(x), pids), [])

    return pids


def _detect_cgroup(pid):
    try:
        with open(os.path.join("/proc", str(pid), "cgroup")) as f:
            data = f.read()
            slot = re.search(":memory:(.*)", data).group(1)
            if re.match('/slots/slot\d+', slot) or re.match('/slots/iss-agent', slot):
                return slot[1:]
    except Exception:
        return None
    return None


def _detect_porto(env):
    if env['type'] == EInstanceTypes.BSCONFIG:
        envconf = '%s:%s' % (env['env']['conf'], env['env']['port'])
    elif env['type'] in (EInstanceTypes.GENCFG, EInstanceTypes.QLOUD, EInstanceTypes.YP):
        envconf = env['env']['containerPath']
    else:
        envconf = None

    conn = porto.Connection(timeout=5)
    try:
        return conn.Find(envconf)
    except Exception:
        return None


def _detect_network_container(cont):
    """Detect network container for specified <cont>

    Suppose we have container <ISS-AGENT--20685/20685_answers_admin_nodejs_testing_lqACjbvvHDO/iss_hook_start> . This container
    always inherit net from parent container, and we have to look in <ISS-AGENT--20685> to understand if we have separate network
    """

    try:
        name = cont.name.partition('/')[0]
        network_cont = cont.conn.Find(name)
        if network_cont.GetData('net') != 'inherited':
            return network_cont
        return None
    except Exception as e:
        return None


def _detect_yp_limits_container(cont):
    """"Detect container with cpu/memory limits/guarantee set"""

    try:
        name = cont.name.partition('/')[0]
        return cont.conn.Find(name)
    except Exception as e:
        return None



def run_command(args, raise_failed=True, timeout=None, sleep_timeout=0.1, cwd=None, close_fds=False, stdin=None):
    """
        Wrapper on subprocess.Popen

        :return (int, str, str): return triplet of (returncode, stdout, stderr)
    """

    if timeout is not None:
        sleep_timeout = min(timeout / 10, sleep_timeout)

    try:
        if stdin is None:
            p = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE, cwd=cwd, close_fds=close_fds)
        else:
            p = subprocess.Popen(args, stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.PIPE, cwd=cwd,
                                 close_fds=close_fds)
    except Exception as e:
        if raise_failed:
            raise Exception("subpocess.Popen for <<%s>> failed: %s" % (args, e))
        else:
            return 1, 'Got unknown error %s' % e, ''

    try:
        if stdin is not None:
            p.stdin.write(stdin)  # Is it correct ?
            p.stdin.close()

        if timeout is None:
            out, err = p.communicate()
            if p.returncode != 0 and raise_failed:
                raise Exception("Command <<%s>> returned %d\nStdout:%s\nStderr:%s" % (args, p.returncode, out, err))
        else:
            out, err = '', ''
            wait_till = time.time() + timeout

            fcntl.fcntl(p.stdout, fcntl.F_SETFL, fcntl.fcntl(p.stdout, fcntl.F_GETFL) | os.O_NONBLOCK)
            fcntl.fcntl(p.stderr, fcntl.F_SETFL, fcntl.fcntl(p.stderr, fcntl.F_GETFL) | os.O_NONBLOCK)

            while time.time() < wait_till:
                p.poll()
                try:
                    while True:
                        r = os.read(p.stdout.fileno(), 1024)
                        out += r
                        if len(r) == 0:
                            break
                except OSError:
                    pass
                try:
                    while True:
                        r = os.read(p.stderr.fileno(), 1024)
                        err += r
                        if len(r) == 0:
                            break
                except OSError:
                    pass
                if p.returncode == 0:
                    return p.returncode, out, err
                if p.returncode is not None:
                    if raise_failed:
                        raise Exception("Command <<%s>> returned %d\nStdout:%s\nStderr:%s" % (args, p.returncode, out, err))
                    else:
                        return p.returncode, out, err
                time.sleep(sleep_timeout)

            if raise_failed:
                raise Exception("Command <<%s>> timed out (%f seconds)" % (args, timeout))
    finally:
        if p.returncode is None:
            p.kill()
            p.wait()

    return p.returncode, out, err


def _is_iss_package_installed():
    code, _, _ = run_command(['dpkg-query', '-W', 'iss-agent'], raise_failed=False)
    if code == 0:
        return True
    if code == 1:
        return False
    return None


def is_iss_installed(
        flag_path=os.path.join(os.environ.get('TMPDIR', '/tmp'), '.instanceusage_iss_installed.flag'),
        threshold_seconds=3600,
):
    if os.uname()[0] != 'Linux':
        return False

    if os.path.exists(flag_path) and (time.time() - os.path.getmtime(flag_path)) < threshold_seconds:
        with open(flag_path, 'r') as f:
            flag = f.read().strip()
            if flag in ('0', '1'):
                return flag == '1'

    iss_installed = _is_iss_package_installed()
    if iss_installed is not None:
        try:
            with open(flag_path, 'w') as f:
                f.write('1' if iss_installed else '0')
        except IOError:
            logging.error('cannot make iss_installed flag')
    return iss_installed


def is_bsconfig_installed():
    return os.path.isfile('/db/bin/bsconfig')


def is_openstack_installed():
    return os.path.isfile('/usr/bin/nova-compute')


def is_iss_up():
    try:
        requests.get('http://localhost:25536/', timeout=5).raise_for_status()
        return True
    except requests.exceptions.RequestException:
        return False


# =====================================================================================
# Instances env
# =====================================================================================

def _update_bsconfig_instance_env(env):
    # fill some log paths
    IKEY = "a_itype_"
    itypes = filter(lambda x: x.startswith(IKEY), env["tags"])
    if len(itypes) == 1:  # Skip old-style instances without autotags
        itype = itypes[0][len(IKEY):]
        if itype == "base":
            env["loadlog"] = "/usr/local/www/logs/current-loadlog-base-%s" % env["port"]
        elif itype == "int":
            env["loadlog"] = "/usr/local/www/logs/current-loadlog-w-int-%s" % env["port"]
        elif itype == "mmeta":
            env["loadlog"] = "/usr/local/www/logs/current-loadlog-base-%s" % env["port"]
        elif itype == "fusion":
            env["loadlog"] = "/usr/local/www/logs/current-loadlog-fusion-%s" % env["port"]

    # fill generaor
    TKEY = "a_topology_version-"
    tags = filter(lambda x: x.startswith(TKEY), env["tags"])
    if len(tags) == 1:  # skip old-style instacnes without autotags
        tag = tags[0][len(TKEY):]
        stable_m = re.match('stable-(\d+)-r(\d+)', tag)
        if stable_m:
            major_tag = int(stable_m.group(1))
            minor_tag = int(stable_m.group(2))
            env["major_tag"] = major_tag
            env["minor_tag"] = minor_tag
            env["revision"] = None
        else:
            env["major_tag"] = 0
            env["minor_tag"] = 0
            trunk_m = re.match('trunk-(\d+)', tag)
            if trunk_m:
                rev = int(trunk_m.group(1))
                env['revision'] = rev

    GKEY = "a_topology_group-"
    group = filter(lambda x: x.startswith(GKEY), env['tags'])
    if len(group) == 1:
        group_name = group[0][len(GKEY):]
        env['group'] = group_name


def get_bsconfig_instances_env():
    retcode, out, err = run_command(["/db/bin/bsconfig", "list", "--json"])
    all_instances = json.loads(out)

    result = {}
    for k in all_instances.keys():
        env = all_instances[k]
        try:
            _update_bsconfig_instance_env(env)

            host = k.split(':')[0]
            port = int(k.split('@')[0].split(':')[1])
            result[(host, port)] = {
                "type": EInstanceTypes.BSCONFIG,
                "env": env,
            }
        except Exception:
            logging.exception("Exception while updating bsconfig instance env")

    return result


def get_iss_instances_env():
    r = requests.get(ISS_ILIST_URL, timeout=10)
    r.raise_for_status()
    all_instances = r.json()

    result = {}
    for k in all_instances.keys():
        env = all_instances[k]

        try:
            env, instance_type = _update_iss_instance_env(env)

            host, _, port = k.partition(':')
            try:
                port = int(port)
            except ValueError:
                # qloud/yp ports
                pass
            result[(host.partition('.')[0], port)] = {
                "type": instance_type,
                "env": env,
            }
        except Exception:
            logging.exception("Exception while updating iss instance env")

    return result


def get_openstack_instances_env():
    insts = get_openstack_instances()
    result = {}

    for uuid, pid in insts:
        try:
            tenant_id = get_tenant_id(uuid)
            result[(socket.gethostname().split('.')[0], uuid)] = {
                'type': EInstanceTypes.OPENSTACK,
                'env': {
                    'pid': pid,
                    'tenant_id': tenant_id,
                }
            }
        except Exception:
            logging.exception("Exception while getting openstack instance")

    return result


def get_fake_instances_env():
    return {
        (socket.gethostname().split('.')[0], 65535): {
            "type": EInstanceTypes.FAKE,
            "env": {"major_tag": 0, "minor_tag": 0},
        },
    }


# =====================================================================================
# FUNCTIONS
# =====================================================================================

class SysBlock(object):
    @staticmethod
    def strip_partition_numbers(device):
        sys_block_devices = os.listdir('/sys/block')
        for parent_device in sys_block_devices:
            if device.startswith(parent_device):
                return parent_device

        return device

    @staticmethod
    def device_exists(device):
        return os.path.exists(os.path.join('/sys/block', device))

    @staticmethod
    def get_existing_device_in_partition_hierarchy(device_short):
        if SysBlock.device_exists(device_short):
            return device_short

        device_short = SysBlock.strip_partition_numbers(device_short)
        if SysBlock.device_exists(device_short):
            return device_short

        return None

    @staticmethod
    def get_slave_devices(device):
        slaves_dir = os.path.join('/sys/block', device, 'slaves')
        return os.listdir(slaves_dir)

    @staticmethod
    def get_major(device):
        with open(os.path.join('/sys/block', device, 'dev')) as dev_file:
            maj = int(dev_file.read().split(':')[0])
            return maj

    @staticmethod
    def is_ssd(device_short):
        device_short = SysBlock.get_existing_device_in_partition_hierarchy(device_short)
        slave_devices = SysBlock.get_slave_devices(device_short)

        # queue/rotational for raids returns 1
        # so we need to check one of the slaves
        # hopefully no one does mixed ssd/hdd raids
        is_raid = bool(slave_devices)
        if is_raid:
            for slave in slave_devices:
                # If slave is unavailable - check the next one
                # It seems like slaves may be missing (not sure why - probably when one of disks dies)
                if SysBlock.get_existing_device_in_partition_hierarchy(slave):
                    return SysBlock.is_ssd(slave)

            raise Exception("Dead raid?")
        else:
            with open(os.path.join('/sys/block', device_short, 'queue/rotational')) as is_rotational_file:
                is_rotational = int(is_rotational_file.read())
                return is_rotational


def get_mounts():
    device_short_to_mount_point = {}
    with open('/proc/mounts') as mounts_file:
        for line in mounts_file:
            device, mount_point = line.split()[:2]
            if not device.startswith('/'):
                continue

            device = os.path.realpath(device)
            device_short = device.split('/')[-1]
            if device_short not in device_short_to_mount_point:
                device_short_to_mount_point[device_short] = mount_point

    return device_short_to_mount_point


def host_ssd_and_hdd_usage(instance_env):
    if instance_env['type'] != EInstanceTypes.FAKE:
        raise Exception("Getting ssd and hdd usage is only available for hosts (and not for individual instances)")

    device_short_to_mount_point = get_mounts()
    MAJ_WHITELIST = [8, 9, 252, 259]  # sd; md (raid); lvm (device-mapper); nvme (blkext)

    hdd_dev_to_mount_point = {}
    ssd_dev_to_mount_point = {}
    for device_short_mounted, mount_point in device_short_to_mount_point.iteritems():
        try:
            device_short = SysBlock.get_existing_device_in_partition_hierarchy(device_short_mounted)
            if device_short is None or SysBlock.get_major(device_short) not in MAJ_WHITELIST:
                continue

            mount_points_dict = hdd_dev_to_mount_point if SysBlock.is_ssd(device_short) else ssd_dev_to_mount_point
            mount_points_dict[device_short_mounted] = mount_point
        except Exception:
            logging.exception("Exception while getting usage for {}".format(device_short_mounted))

    def get_usage_and_total(mount_points):
        used = 0
        total = 0
        for mp in mount_points:
            stats = os.statvfs(mp)
            total += stats.f_frsize * stats.f_blocks
            used += stats.f_frsize * (stats.f_blocks - stats.f_bfree)

        return used, total

    ssd_used, ssd_total = get_usage_and_total(ssd_dev_to_mount_point.itervalues())
    hdd_used, hdd_total = get_usage_and_total(hdd_dev_to_mount_point.itervalues())

    return {
        'ssd_usage': ssd_used,
        'ssd_total': ssd_total,
        'hdd_usage': hdd_used,
        'hdd_total': hdd_total,
    }


def instance_cpu_and_net_usage(instance_env, xparams):
    # get some system params
    HOST_OS = os.uname()[0]
    HZ = os.sysconf('SC_CLK_TCK')
    NCPU = _host_ncpu()
    TIMEOUT = xparams.get('instance_cpu_usage_timeout', 5)

    if instance_env["type"] != EInstanceTypes.FAKE:
        # detect pids
        pids = _detect_pids(instance_env)

        # meausure, wait timeout, measure again
        if HOST_OS == 'Linux':
            # check if process in porto
            porto_container = _detect_porto(instance_env)

            # find porto
            porto_network_container = _detect_network_container(porto_container)

            if porto_container is not None:
                def parse_cpu_data(out):
                    return float(out) / 1000 / 1000 / 1000

                def parse_net_data(out):
                    fields = [kv.strip().split(': ') for kv in out.split(';')]
                    return {k: int(v) for (k, v) in fields}

                def get_bytes(net_data):
                    return sum(net_data.itervalues())

                out_cpu = porto_container.GetData('cpu_usage')
                if porto_network_container:
                    before_net_tx = get_bytes(parse_net_data(porto_network_container.GetData('net_tx_bytes')))
                    before_net_rx = get_bytes(parse_net_data(porto_network_container.GetData('net_rx_bytes')))
                else:
                    before_net_tx = 0
                    before_net_rx = 0

                beforet = monoTime()

                before_cpu = parse_cpu_data(out_cpu)

                time.sleep(TIMEOUT)

                out_cpu = porto_container.GetData('cpu_usage')
                if porto_network_container:
                    after_net_tx = get_bytes(parse_net_data(porto_network_container.GetData('net_tx_bytes')))
                    after_net_rx = get_bytes(parse_net_data(porto_network_container.GetData('net_rx_bytes')))
                else:
                    after_net_tx = 0
                    after_net_rx = 0

                aftert = monoTime()

                after_cpu = parse_cpu_data(out_cpu)
                deltat = aftert - beforet

                cpu_usage = max(0., (after_cpu - before_cpu) / NCPU / deltat)
                net_tx = int((after_net_tx - before_net_tx) / deltat)
                net_rx = int((after_net_rx - before_net_rx) / deltat)
                res = {
                    'instance_cpu_usage': cpu_usage,
                    'instance_net_tx_usage': net_tx,
                    'instance_net_rx_usage': net_rx,
                }

                return res

            # go default stats
            def cpustat(pid):
                try:
                    return open(os.path.join('/proc', '%s' % pid, 'stat')).read().split()
                except Exception:
                    return None

            before = map(cpustat, pids)
            beforet = monoTime()

            time.sleep(TIMEOUT)

            after = map(cpustat, pids)
            aftert = monoTime()

            result = 0.0
            for i in range(len(before)):
                if before[i] is not None and after[i] is not None:
                    result += (float(after[i][13]) - float(before[i][13]) + float(after[i][14]) - float(
                        before[i][14])) / (aftert - beforet) / HZ / NCPU
            return {'instance_cpu_usage': result}
        elif HOST_OS == 'FreeBSD':
            def _convert(text):
                text = text.split('\n')[1]
                minutes, _, seconds = text.partition(':')
                seconds = re.sub(',', '.', seconds)  # Issue with russian locale
                return int(minutes) * 60 + float(seconds)

            beforet = monoTime()
            before = map(lambda pid: run_command(["ps", "-o", "time", str(pid)])[1], pids)

            time.sleep(TIMEOUT)

            aftert = monoTime()
            after = map(lambda pid: run_command(["ps", "-o", "time", str(pid)])[1], pids)

            return {'instance_cpu_usage': sum(map(lambda b, a: (_convert(a) - _convert(b)) / (aftert - beforet) / NCPU, before, after))}
        else:
            raise Exception("Calculating instance cpu usage is not supported in os %s" % HOST_OS)
    else:  # fake instance cpu usage == all processes instance cpu usage
        if HOST_OS == 'Linux':
            def parse_cpu_data(proc_stat_first_line):
                return map(lambda x: int(x), re.search('cpu +(.*)', proc_stat_first_line).group(1).split(' '))

            def get_total_bytes(proc_net_dev_lines):
                body = proc_net_dev_lines[2:]
                splitted_lines = [filter(None, line.strip().split(' ')) for line in body]

                net_tx_total = 0
                net_rx_total = 0
                for fields in splitted_lines:
                    if not fields[0].startswith('eth'):
                        continue
                    net_rx = int(fields[1])
                    net_tx = int(fields[9])

                    net_rx_total += net_rx
                    net_tx_total += net_tx

                return net_rx_total, net_tx_total

            before_cpu = parse_cpu_data(open('/proc/stat').readline())
            before_net_rx, before_net_tx = get_total_bytes(open('/proc/net/dev').readlines())
            beforet = monoTime()

            time.sleep(TIMEOUT)

            after_cpu = parse_cpu_data(open('/proc/stat').readline())
            after_net_rx, after_net_tx = get_total_bytes(open('/proc/net/dev').readlines())
            aftert = monoTime()

            idle = after_cpu[3] - before_cpu[3]
            tot = sum(after_cpu) - sum(before_cpu)

            cpu_usage = (tot - idle) / float(tot)

            deltat = aftert - beforet

            net_tx = int((after_net_tx - before_net_tx) / deltat)
            net_rx = int((after_net_rx - before_net_rx) / deltat)

            return {
                'instance_cpu_usage': cpu_usage,
                'instance_net_tx_usage': net_tx,
                'instance_net_rx_usage': net_rx
            }
        elif HOST_OS == 'FreeBSD':
            status, before, _ = run_command(["sysctl", "-n", "kern.cp_time"])
            before = map(lambda x: int(x), before.split())

            time.sleep(TIMEOUT)

            status, after, _ = run_command(["sysctl", "-n", "kern.cp_time"])
            after = map(lambda x: int(x), after.split())

            idle = after[4] - before[4]
            tot = sum(after) - sum(before)
            return (tot - idle) / float(tot)
        else:
            raise Exception("Unkown os %s" % HOST_OS)


def instance_mem_usage(instance_env, xparams):
    del xparams
    CGROUP_MEM_ROOT = "/sys/fs/cgroup/memory"
    HOST_OS = os.uname()[0]

    if instance_env["type"] != EInstanceTypes.FAKE:
        pids = _detect_pids(instance_env)

        if HOST_OS == "Linux":
            # check if process in porto
            porto_container = _detect_porto(instance_env)
            if porto_container is not None:
                out = porto_container.GetData('memory_usage')
                return float(out) / 1024 / 1024 / 1024

            # check if processes in cgroup
            cgroups = list(set(filter(lambda x: x is not None, map(_detect_cgroup, pids))))
            if len(cgroups) == 1:  # found exactly one cgroup
                return float(open(
                    os.path.join(CGROUP_MEM_ROOT, cgroups[0], "memory.usage_in_bytes")).read()) / 1024 / 1024 / 1024

            # go default stats
            if 'tags' in instance_env['env'] and \
               'a_ctype_hamster' in instance_env['env']['tags']:
                # hamster should be treated separately
                def memstat(pid):
                    try:
                        fname = os.path.join('/proc', '%s' % pid, 'statm')
                        with file(fname) as f:
                            data = f.read()

                        pages = int(data.split()[1]) - int(data.split()[2])

                        return float(pages) * 4096 / 1024 / 1024 / 1024
                    except Exception:
                        raise

                return sum(map(lambda x: memstat(x), pids))

            else:
                # we can not calculate memory usage precisely without cgroup so suppose all VmLib memory
                # is common for all instances
                def memstat(pid):
                    try:
                        fname = os.path.join('/proc', '%s' % pid, 'status')
                        with file(fname) as f:
                            data = f.read()

                        rss = float(re.search("VmRSS:\s+(\d+)", data).group(1)) / 1024 / 1024
                        lib = float(re.search("VmLib:\s+(\d+)", data).group(1)) / 1024 / 1024

                        return rss, lib
                    except Exception:
                        return 0., 0.

                rr = map(lambda x: memstat(x), pids)

                if len(rr):
                    total_rss = sum(map(lambda x: x[0], rr))
                    total_lib = sum(map(lambda x: min(x[0], x[1]), rr))
                    max_lib = max(map(lambda x: x[1], rr))
                    return total_rss - total_lib + max_lib
                else:
                    return 0.

        elif HOST_OS == "FreeBSD":
            if len(pids) == 0:
                return 0.0

            status, out, err = run_command(["ps", "-o", "rss", "-p", ','.join(map(lambda x: str(x), pids))])

            total = 0
            for line in out.split("\n"):
                try:
                    total += int(line)
                except ValueError:
                    pass

            return float(total) / 1024 / 1024
        else:
            raise Exception("Calculating instance mem usage is not supported in os %s" % HOST_OS)
    else:  # fake instance memory usage == all memory - free memory
        if HOST_OS == "Linux":
            with open("/proc/meminfo") as f:
                data = f.read()
                return (float(re.search("MemTotal:\s+(\d+)", data).group(1)) - float(
                    re.search("MemFree:\s+(\d+)", data).group(1))) / 1024 / 1024
        elif HOST_OS == "FreeBSD":
            return 0.  # FIXME: not implemented
        else:
            raise Exception("Calculating instance mem usage is not supported in os %s" % HOST_OS)


def instance_anon_mem_usage(porto_conn, instance_data):
    try:
        container_name = instance_data.get('env', dict()).get('containerPath', None)
        if container_name is None:
            return 0

        return float(porto_conn.Find(container_name).GetData('anon_usage')) / 1024 / 1024 / 1024
    except Exception:
        return 0


# =================================== RX-552 START ===============================================
def instance_mem_guarantee(porto_conn, instance_data):
    try:
        container_name = instance_data.get('env', dict()).get('containerPath', None)
        if container_name is None:
            return 0

        container = porto_conn.Find(container_name)
        if not container:
            return 0

        guarantee_container = _detect_yp_limits_container(container)

        return float(guarantee_container.GetData('memory_guarantee')) / 1024 / 1024 / 1024
    except Exception:
        return 0

def instance_cpu_guarantee(porto_conn, instance_data):
    try:
        container_name = instance_data.get('env', dict()).get('containerPath', None)
        if container_name is None:
            return 0

        container = porto_conn.Find(container_name)
        if not container:
            return 0

        guarantee_container = _detect_yp_limits_container(container)

        return float(guarantee_container.GetData('cpu_guarantee')[:-1]) / _host_ncpu()
    except Exception:
        return 0
# =================================== RX-522 FINISH ==============================================



# ===============================================================================
# main stuff
# ===============================================================================

def generate(arg):
    signal.alarm(300)
    (host, port), instance_env = arg

    if os.path.exists('/run/portod.socket'):
        c = porto.Connection(timeout=5)
        c.connect()
    else:
        c = None

    try:
        instance_data = {
            'port': port,
            'ts': time.time(),
            'instance_mem_usage': instance_mem_usage(instance_env, {}),
            'instance_anon_mem_usage': instance_anon_mem_usage(c, instance_env),
            'instance_mem_guarantee': instance_mem_guarantee(c, instance_env),
        }

        if instance_env['type'] == EInstanceTypes.FAKE:
            instance_data.update(host_ssd_and_hdd_usage(instance_env))
            instance_data['type'] = 'host'
        elif instance_env['type'] == EInstanceTypes.OPENSTACK:
            instance_data['tenant_id'] = instance_env['env']['tenant_id']
            instance_data['type'] = 'openstack'
        elif instance_env['type'] == EInstanceTypes.QLOUD:
            for param in ('service', 'environ', 'app', 'project', 'install_type'):
                instance_data[param] = instance_env['env'][param]
            instance_data['type'] = 'qloud'
        elif instance_env['type'] == EInstanceTypes.GENCFG:
            instance_data.update({
                'major_tag': instance_major_tag(instance_env),
                'minor_tag': instance_minor_tag(instance_env),
                'version': instance_version(instance_env),
                'group': instance_group(instance_env),
                'type': 'iss'
            })
        elif instance_env['type'] == EInstanceTypes.YP:
            instance_data.update({
                'group': instance_env['env']['group'],
                'type': EInstanceTypes.YP,
                'instance_cpu_guarantee': instance_cpu_guarantee(c, instance_env),
            })

        signals_and_usages = instance_cpu_and_net_usage(instance_env, {})
        if signals_and_usages:
            instance_data.update(signals_and_usages)

        return instance_data
    except Exception:
        logging.exception("generate")
        return None


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-f', '--format', type=str, default='pretty',
                        choices=('pretty', 'binary'),
                        help='Output in serialized or pretty format')
    parser.add_argument('--search', action='store_true', default=False,
                        help='Runs in yandex-search environment with bsconfig, iss, and stuff.')
    return parser.parse_args()


def main():
    args = parse_args()

    all_env = get_fake_instances_env()

    if args.search or (is_iss_installed() and is_iss_up()):
        try:
            all_env.update(get_iss_instances_env())
        except Exception:
            logging.exception("get_iss_instances failed")

    if is_bsconfig_installed():
        try:
            all_env.update(get_bsconfig_instances_env())
        except Exception:
            logging.exception("get_bsconfig_instances failed")

    if not args.search and is_openstack_installed():
        try:
            all_env.update(get_openstack_instances_env())
        except Exception:
            logging.exception("get_openstack_instances failed")

    p = multiprocessing.Pool(multiprocessing.cpu_count())
    instances_data = p.map(generate, all_env.items())
    instances_data = filter(lambda x: x is not None, instances_data)

    if args.format == 'pretty':
        print(json.dumps(instances_data, sort_keys=True, indent=4))
    elif args.format == 'binary':
        print(msgpack.packb(instances_data))


if __name__ == '__main__':
    main()
