from __future__ import absolute_import, print_function, division

import argparse
import msgpack
import py
import os
import datetime
import shlex
import time
import subprocess
from threading import Timer
from collections import namedtuple

from pprint import pprint


REPORT_VERSION = 2

CommandResult = namedtuple('CommandResult', ['returncode', 'out', 'err', 'has_timeout', 'elapsed'])


def kill_process(process, dto):
    """
    timed out recipe from
      https://stackoverflow.com/questions/1191374/using-module-subprocess-with-timeout/10768774#10768774
    """
    dto["value"] = True
    process.kill()


def run_command(args, lines=False, timeout_sec=60, exception_on_timeout=True, update_env=None):
    if type(args) == str:
        args = shlex.split(args)

    cmdline = " ".join(args)

    started_time = time.time()

    new_env = os.environ.copy()
    if update_env:
        new_env.update(**update_env)

    proc = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE, env=new_env)

    timeout_dto = {"value": False}
    timer = Timer(timeout_sec, kill_process, [proc, timeout_dto])

    timer.start()
    out, err = proc.communicate()
    timer.cancel()

    if lines:
        out = filter(None, out.splitlines())
        err = filter(None, err.splitlines())

    elapsed_time = time.time() - started_time

    if exception_on_timeout and timeout_dto["value"]:
        raise Exception("got timeout (%r sec) on [%s]" % (timeout_sec, cmdline))

    return CommandResult(returncode=proc.returncode, out=out, err=err, has_timeout=timeout_dto["value"],
                         elapsed=elapsed_time)


def get_user_sessions():
    """
    Example command output:
     ~ who
     i-dyachkov pts/0        2021-12-22 14:21 (2a02:6b8:81:1:2cf0:11d0:13ac:f93e)
     i-dyachkov pts/9        2021-12-22 16:27 (2a02:6b8:81:1:2cf0:11d0:13ac:f93e)
    """
    result = {}
    for s in run_command('who', lines=True).out:
        ss = s.strip().split()
        if len(ss) > 1:
            login = ss[0]
            result[login] = datetime.datetime.utcnow().isoformat() + 'Z'
    return result


def get_user_actions():
    result = {}
    for s in run_command('w -h', lines=True, update_env={'PROCPS_USERLEN': '32'}).out:
        ss = s.strip().split(' ', 1)
        login = ss[0]
        if login not in result:
            result[login] = []

        if len(result[login]) < 10:
            result[login].append(ss[1].strip())

    return result


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-f', '--format', choices=('pretty', 'msgpack'), default='pretty')
    return parser.parse_args()


def main():
    args = parse_args()

    result = {
        'report_version': REPORT_VERSION,
        'user_sessions': get_user_sessions(),
        'user_actions': get_user_actions()
    }

    if args.format == 'pretty':
        pprint(result)
    if args.format == 'msgpack':
        print(msgpack.packb(result))


if __name__ == '__main__':
    main()
