#!/usr/bin/env bash

DEPENDENCIES="api kernel library"

set -e

require_python()
{
    if [ -z "${PYTHON}" ]; then
        echo "Please export \$PYTHON variable!"
        exit 1
    else
        true
    fi
}

do_info()
{
    echo " * " $*
}

do_cleanup()
{
    do_info "cleanup $*"

    rm -rf ./dist ./*.egg-info

    if [ "$1" = "full" ]; then
        rm -rf ./build ./snapshot
    fi
}

do_bin_snapshot()
{
    do_info "adding binary files"
    for f in $(ls ./bin); do
        if [ "$1" = "symlink" ]; then
            ln -s ../../bin/$f ./snapshot/bin
        elif [ "$1" = "link" ]; then
            cp --link ./bin/$f ./snapshot/bin
        else
            cp ./bin/$f ./snapshot/bin
        fi
    done
}

do_prepare()
{
    do_info "preparing directories"
    mkdir -p ./snapshot/lib ./snapshot/bin
}

do_ctl()
{
    do_info "adding heartbeat-server.scsd"

    if [ "$1" = "symlink" ]; then
        ln -sf ../heartbeat-server.scsd snapshot
    else
        cp -f ./heartbeat-server.scsd snapshot
    fi

    $PYTHON -m compileall -q snapshot
}

do_call_setuppy()
{
    do_info "running: setup.py $*"
    PYTHONPATH=$PYTHONPATH:./snapshot/lib $PYTHON setup.py -q "$@"
}

do_bundle()
{
    require_python
    do_cleanup full
    do_prepare
    do_ctl
    do_call_setuppy egg_info bdist_egg
    do_call_setuppy easy_install -d ./snapshot/lib -s ./snapshot/bin -Z -O0 -m dist/*.egg

    mkdir -p snapshot/bin/plugins
    for path in $(ls ./snapshot/bin); do
        cut=${path##plugin__}
        if [ "$path" != "$cut" ]; then
            mv snapshot/bin/$path snapshot/bin/plugins/$cut
        fi
    done

    base=""
    libname=$(ls snapshot/lib | head -n1)

    for path in ya skynet services heartbeatserver; do
        if [ "$base" != "" ]; then
            path="$base/$path"
        fi
        base=$path

        if [ ! -f snapshot/lib/$libname/$path/__init__.py ]; then
            echo "Creating __init__.py for $path"
            echo "__import__(\"pkg_resources\").declare_namespace(__name__)" >  snapshot/lib/$libname/$path/__init__.py
        fi
    done

    # snapshot/lib/skynet_heartbeat_server_service-0.1-py2.7.egg/ya

    do_cleanup
}

do_develop()
{
    require_python
    do_cleanup
    rm -rf ./snapshot

    do_prepare
    do_ctl symlink
    do_call_setuppy develop -d ./snapshot/lib -s ./snapshot/bin -z -O2 -m

    rm -rf snapshot/ve
    for mod in ya ya/skynet ya/skynet/services; do
        mkdir -p snapshot/ve/${mod}
        echo '__import__("pkg_resources").declare_namespace(__name__)' > snapshot/ve/${mod}/__init__.py
    done

    local _egg_info=$(ls -d *.egg-info)
    mv ${_egg_info} snapshot/ve

    local _egg_link=$(basename $(ls snapshot/lib/*.egg-link))
    echo "$(pwd)/snapshot/ve" > snapshot/lib/${_egg_link}
    echo "../.." >> snapshot/lib/${_egg_link}

    ln -s ../../../../../src ./snapshot/ve/ya/skynet/services/heartbeatserver
    do_cleanup
}

do_pylint()
{
    require_python
    PYTHONPATH=./snapshot/lib:$PYTHONPATH $PYTHON $(dirname $PYTHON)/pylint --rcfile=./pylint.conf ya.skynet.services.heartbeatserver "$@"
}

do_test()
{
    require_python
    mkdir -p tmp
    do_call_setuppy test "$@"
    do_cleanup
}

do_shell()
{
    require_python
    do_prepare
    cat > snapshot/shell <<END
#!/usr/bin/env python
from pkg_resources import require, load_entry_point
require('skynet-heartbeat-server-service')
raise SystemExit(load_entry_point('ipython', 'console_scripts', 'ipython')())
END
    chmod +x snapshot/shell
    PYTHONPATH=./snapshot/lib:$PYTHONPATH $PYTHON snapshot/shell
}

do_run()
{
    PYTHONPATH=./snapshot/lib:$PYTHONPATH $PYTHON ./snapshot/bin/daemon \
        -C AppPath=$PWD/snapshot \
        -C SupervisorPath=$PWD/snapshot \
        -c share/config.yaml "$@"
}

do_tgz()
{
    do_bundle
    rm -rf heartbeat-server.tgz
    tar --exclude '.svn' -C snapshot -czvf heartbeat-server.tgz .
}

case "$1" in
    bundl*)
        do_bundle
        ;;

    dev*)
        do_develop
        ;;

    clean*)
        do_cleanup full
        ;;

    pylint*)
        shift
        do_pylint "$@"
        ;;

    test*)
	shift
        do_test "$@"
        ;;

    shell*)
        do_shell
        ;;

    run*)
        shift 1
        do_run "$@"
        ;;
    tgz*)
        do_tgz
        ;;
    dep*)
        echo ${DEPENDENCIES}
        ;;

    *)
        echo "Usage $0 [options] develop | bundle | clean | pylint | shell | run | tgz | deps | help"
        ;;
esac
