from __future__ import absolute_import

import os
import time
import datetime
import traceback as tb

import py
import yaml
import gevent
import gevent.socket
import gevent.monkey

import pymongo

# Make pymongo compatible with gevent (see http://api.mongodb.org/python/current/examples/gevent.html for details).
pymongo.pool.socket = gevent.socket

from .. import utils
from ..daemon import config
from ..utils.triggers import Trigger

from . import plugins


class UnsupportedReport(Exception):
    def __init__(self, src, name):
        super(UnsupportedReport, self).__init__('Unsupported report from %r of type %r' % (src, name))


class Brigadier(object):
    # Period in seconds in which connected hosts are considered as active.
    COLLECT_PERIOD = 3600
    # Per-plugin configuration parameters, which will be passed from the root configuration.
    PER_PLUGIN_CFG_PARAMS = (
        'restart_pause greetings_wait start_attempts discard_restarts fatal_restarts '
        'processing_limit queue_data_limit queue_limits_check'
    ).split()

    def __init__(self, ctx):
        super(Brigadier, self).__init__()

        self.ctx = ctx
        self.cfg = ctx.cfg.bulldozer
        self.rootLog = ctx.log.getChild('bulldozer')
        self.log = self.rootLog.getChild('brigadier')

        # Statistics
        self.triggers = {}
        self.plugins = {}
        self.clients = {}
        self.started = 0

        # Database
        self.conn = None
        self.hostinfo = None
        self.dbURI = self.ctx.cfg.database.uri
        self.log.info('Database URI: %r', self.dbURI)

        # Lacmus
        self.lacmus = self.ctx.cfg.get('lacmus', {})
        self.log.info('Lacmus: [%r, %r] %r',
                      self.lacmus.get('timeout', None),
                      self.lacmus.get('retries', None),
                      self.lacmus.get('uri', None))

    def start(self):
        # First of all, connect to MongoDB
        self.conn = pymongo.mongo_client.MongoClient(self.dbURI, use_greenlets=True, max_pool_size=10)
        self.hostinfo = self.conn[utils.dbName(self.dbURI)]['hostinfo']

        # Start `PluginManager`'s scheduler
        plugins.Supervisor._broker.start(self.rootLog)

        # Load user plugin(s) configuration files and start them.
        cfg = self.cfg.plugins.user
        cfgdir = py.path.local(cfg.cfgdir)
        self.log.info('Loading user plugins group at %r.', cfgdir.strpath)
        if cfgdir.check(exists=1):
            for f in cfgdir.listdir():
                if not f.check(file=1) or f.ext != '.conf':
                    continue
                plugin_config = yaml.load(open(f.strpath, 'rb').read())
                plugin_name = f.purebasename.lower()

                # Use <name>.py executable by default
                if 'executable' not in plugin_config:
                    plugin_config['executable'] = plugin_name + '.py'

                # Update with absolute path if not already set
                plugin_config['executable'] = os.path.join(
                    cfg.bindir, plugin_config['executable']
                )

                self._startPlugin(
                    plugin_name,
                    plugin_config,
                    source='file %s' % (f, )
                )

        self.log.info('Loading system plugins group.')

        aliases = {}

        for plugin_name, plugin_config in self.cfg.plugins.system.cfgs.items():
            alias = plugin_config.get('alias')
            if alias is not None:
                aliases.setdefault(alias, []).append(plugin_name)

        for plugin_name, plugin_config in self.cfg.plugins.system.cfgs.items():
            if 'alias' in plugin_config:
                continue

            if plugin_name in aliases:
                plugin_config.setdefault('aliases', []).extend(aliases[plugin_name])

            if 'executable' not in plugin_config:
                plugin_config['executable'] = plugin_name + '.py'

            plugin_config['executable'] = os.path.join(
                self.cfg.plugins.system.bindir,
                plugin_config['executable'],
            )

            self._startPlugin(
                plugin_name,
                plugin_config,
                source='system config (name=%s)' % (plugin_name, )
            )

        # Validate redirects
        for name in self.plugins.keys():
            p = self.plugins[name]
            if isinstance(p, str):
                if p not in self.plugins:
                    del self.plugins[name]
                    self.log.warn('No destination plugin for redirect from %r to %r', name, p)

        # Also, start self state reporter
        self.stateReporter = gevent.spawn(self._reporter)
        self.started = time.time()

        return self

    def stop(self):
        for name, p in self.plugins.iteritems():
            if not isinstance(p, str):
                self.log.info('Stopping plugin %r.', name)
                p.stop()

        # Handle reporter
        self.stateReporter.kill(gevent.GreenletExit)
        self.stateReporter.join()

        # Finally, stop `PluginManager`'s scheduler
        plugins.Supervisor._broker.stop()

    def on_contact(self, src):
        """
        Updates host's last contact timestamp.
        :param src: Name of contacted host.
        :return: None
        """
        self.hostinfo.update({'host': src}, {'$set': {'last_update': datetime.datetime.now()}})

    def process(self, src, report, expires=float('inf'), cb=None):
        """
        Processes the given report of given type.
        :param src: Source host name.
        :param report: Report content. Report name should be available as 'name' entry of this dictionary.
        :param expires: Timestamp, when the report goes to be expired and can be dropped out of the queue.
        :param cb: Callback to send task result with.
        :return: `plugins.Supervisor.process()` method call result.
        """
        self.clients[src] = time.time()
        name = report['name'].lower()
        p = self.plugins.get(name)
        if not p:
            raise UnsupportedReport(src, name)
        if isinstance(p, str):
            name = p
            p = self.plugins[name]

        return p.process(src, name, report, expires, cb)

    def status(self):
        now = time.time()
        uptime = now - self.started
        # In case of uptime is too low, postpone reply for some time
        if uptime < self.cfg.reporter.period / 3:
            sleep = self.cfg.reporter.period / 3 - uptime
            self.log.info('Too low uptime. Postponing status reply for %.0f seconds.', sleep)
            gevent.sleep(sleep)
            return self.status()

        # Secondly, drop out outdated clients from the collected statistics
        self.clients = dict(filter(lambda (h, ts): ts + self.COLLECT_PERIOD >= now, self.clients.iteritems()))

        # Report active clients count and uptime in seconds.
        ret = {
            'clients': len(self.clients),
            'uptime': uptime,
            'plugins': {}
        }

        # Now collect per-plugin statistics
        pstat = ret['plugins']
        for name, p in self.plugins.iteritems():
            if isinstance(p, str):
                pstat[name] = {'redirect': p}
            elif p.aliases[-1] == name:
                ps = pstat[name] = p.status()
                if len(p.aliases) > 1:
                    ps['aliases'] = filter(lambda x: x != name, p.aliases)

                state = ps['state'] = {}
                for item, triggers in self.triggers[name].iteritems():
                    st = map(lambda tr: tr.__repr__(ps[item]), filter(lambda tr: tr.check(ps[item]), triggers))
                    state[item] = ', '.join(st) if st else 'OK'

        return ret

    def ping(self):
        pong = self.plugins[self.cfg.reporter.name].status()['instances'] > 0
        self.log.info('Ping result: %r', pong)
        return pong

    def _startPlugin(self, name, plugin_config, source):
        self.log.info('Loading %r plugin (source: %s)', name, source)
        cfg = config.AppConfig()
        appCfg = self.ctx.cfg

        cfg.update({
            'instances': 1,
            'arguments': [],
            'aliases': [],
            'triggers': {},
            'database_uri': self.dbURI,
            'lacmus_uri': self.lacmus.get('uri', None),
            'lacmus_timeout': self.lacmus.get('timeout', None),
            'lacmus_retries': self.lacmus.get('retries', None),
            'report_period': self.cfg.reporter.period,
            'application_root': appCfg.AppPath,
        })
        cfg.update(dict([(n, getattr(self.cfg, n), ) for n in self.PER_PLUGIN_CFG_PARAMS]))
        cfg.load(plugin_config)
        redirect = getattr(cfg, 'redirect', None)
        if redirect:
            self.log.info('Redirect from %r to %r added.', name, redirect)
            self.plugins[name] = redirect
            return

        triggers = {}
        for n in self.cfg.triggers:
            triggers[n] = getattr(self.cfg.triggers, n) if n not in cfg.triggers else getattr(cfg.triggers, n)
        try:
            t = self.triggers[name] = {}
            for n, v in triggers.items():
                t[n] = Trigger.fromString(v)
            self.log.info('Plugin %r triggers: %r', name, t)
            plugin = plugins.Supervisor(name, cfg, self.rootLog.getChild('plugin').getChild(name))
            self.plugins.update(dict.fromkeys(plugin.aliases, plugin))
            plugin.start()
        except Exception as ex:
            self.log.critical('Unable to load plugin %r: %s', name, repr(ex))

    def _reporter(self):
        cfg = self.cfg.reporter
        host = gevent.socket.gethostname()
        while True:
            gevent.sleep(cfg.period)
            try:
                st = self.status()
                self.log.debug('Bulldozer status: %r', st)
                self.process(host, {'report': st, 'name': cfg.name}, expires=time.time() + 30)
            except Exception:
                self.log.fatal(
                    'Shutting down because of exception caught on capturing own status. %s',
                    tb.format_exc()
                )
                os._exit(-1)  # KORUM: TODO: Any other way to stop own service?
