from __future__ import print_function

import os
import sys
import fcntl
import select
import datetime

import pkg_resources

pkg_resources.require('pymongo')
pkg_resources.require('msgpack-python')

import pymongo
import msgpack
import time
import requests

from .. import utils


class LacmusReporter(object):
    """
    Lacmus report helper.
    """
    def __init__(self, lacmus_uri=None, log=None, acc_timeout=10):
        self.lacmusURI = lacmus_uri if lacmus_uri else os.getenv('LACMUS_URI')
        if not self.lacmusURI:
            return

        self.log = log
        self.timeout = int(os.getenv('LACMUS_TIMEOUT', '10'))
        self.max_retries = int(os.getenv('LACMUS_RETRIES', '10'))
        self.acc_timeout = acc_timeout

        self.cur_report = {'reports': {}}
        self.acc_reports = []
        self.left_retries = self.max_retries
        self.next_send_time = time.time() + self.acc_timeout

    def add_signal(self, name, value):
        if not self.lacmusURI or not value:
            # lacmusUri is not specified or values are empty -> ignore it
            return

        if not isinstance(value, str):
            # try to convert non-string values to string
            if isinstance(value, list) or isinstance(value, tuple):
                value = '.'.join(map(str, value))
            else:
                raise Exception('Unexpected type of signal value')

        self.cur_report['reports'][name] = value

    def send_report(self, hostname, timestamp):
        if not self.lacmusURI:
            # lacmusUri is not specified -> do nothing
            return

        if len(self.cur_report['reports']):
            # append report if it is not empty
            self.cur_report['host'] = hostname
            self.cur_report['timestamp'] = timestamp
            self.acc_reports.append(self.cur_report)
            self.cur_report = {'reports': {}}

        if time.time() >= self.next_send_time:
            self._send_internal()
            self.next_send_time = time.time() + self.acc_timeout

    def _send_internal(self):
        if not len(self.acc_reports):
            # nothing to send
            if self.log:
                self.log.debug('lacmus: nothing to send')
            return

        if self.log:
            self.log.debug('lacmus: sending report:\n{}'.format('\n'.join(str(rep) for rep in self.acc_reports)))

        data = msgpack.dumps(self.acc_reports)

        headers = {
            'User-agent': 'Heartbeat/1.0',
            'Host': 'lacmus.yandex-team.ru'
        }

        try:
            resp = requests.post(self.lacmusURI, headers=headers, data=data, timeout=self.timeout)
            if resp.status_code == requests.codes.ok:
                self.left_retries = 0
            else:
                if self.log:
                    self.log.warning('lacmus: failed to send: %d', resp.status_code)
                self.left_retries -= 1
        except BaseException as ex:
            self.left_retries -= 1
            if self.log:
                self.log.warning('lacmus: failed to send: %r', ex)

        if not self.left_retries:
            # we have sent report successfully or have no more tries
            # remove all accumulated reports
            self.acc_reports = []
            self.left_retries = self.max_retries


class ReportDatabase(object):
    """
    Database accessor helper.
    """

    def __init__(self, db_uri=None):
        self.dbURI = db_uri if db_uri else os.getenv('DATABASE_URI')
        if not self.dbURI:
            raise Exception("Database URI wasn't passed via `DATABASE_URI` environment variable.")
        self.conn = pymongo.mongo_client.MongoClient(self.dbURI, max_pool_size=5)
        self.db = self.conn[utils.dbName(self.dbURI)]

    def __call__(self):
        return self.db


class ReportCollection(object):
    """
    Database collection accessor helper.
    """

    def __init__(self, name, rdb=None, onCreate=None, onUpdate=None):
        self.db = ReportDatabase() if not rdb else rdb
        db = self.db()

        if onCreate or onUpdate:
            if onCreate and name not in db.collection_names():
                self.coll = onCreate(db, name)
            else:
                self.coll = db[name]
                if onUpdate:
                    onUpdate(self.coll)

        else:
            self.coll = db[name]
            self.coll.ensure_index('host', unique=True)

        self.bulk = None
        self.bulk_ts = None

    def __call__(self):
        return self.coll

    def _bulk_init(self, step):
        if self.bulk:
            if int(time.time() // step) != self.bulk_ts:
                try:
                    self.bulk.execute()
                finally:
                    self.bulk = None

        if not self.bulk:
            self.bulk = self.coll.initialize_unordered_bulk_op()
            self.bulk_ts = int(time.time() // step)

    def insert(self, report, bulk=None):
        if not bulk:
            return self.coll.insert(report)
        else:
            self._bulk_init(bulk)
            self.bulk.insert(report)

    def update(self, query, report, remove=None, bulk=None, replace=False):
        if not isinstance(report, dict):
            raise TypeError('Report is not a `dict` instance.')

        if bulk:
            self._bulk_init(bulk)

        # remove specified fields
        if remove:
            if not isinstance(remove, dict):
                raise TypeError('Remove is not a `dict` instance.')

            if bulk:
                self.bulk.find({'host': query} if isinstance(query, str) else query).update({'$unset': remove})
            else:
                self.coll.update(
                    {'host': query} if isinstance(query, str) else query,
                    {'$unset': remove}
                )

        # update fields
        report['last_update'] = datetime.datetime.now()
        if replace:  # pymongo 2.7.2 doesn't have non-bulk replace_one
            if isinstance(query, str):
                report['host'] = query
            self.bulk.find(
                {'host': query} if isinstance(query, str) else query
            ).upsert().replace_one(
                report
            )
        elif bulk:
            self.bulk.find(
                {'host': query} if isinstance(query, str) else query
            ).upsert().update(
                {'$set': report}
            )
        else:
            self.coll.update(
                {'host': query} if isinstance(query, str) else query,
                {'$set': report},
                upsert=True
            )


class Communicator(object):
    """
    Heartbeat server-to-bulldozer communication helper.
    """

    def __init__(self):
        self.istream = sys.stdin
        self.ostream = os.fdopen(os.dup(sys.stdout.fileno()), 'wb', 0)
        sys.stdout.close()  # Be a paranoid - avoid mistyped `print()`s.

        # Make input stream non-blocking
        fd = self.istream.fileno()
        fcntl.fcntl(fd, fcntl.F_SETFL, fcntl.fcntl(fd, fcntl.F_GETFL) | os.O_NONBLOCK)

    def ready(self):
        self.ostream.write("\n")
        # No need to flush here because the interpreter should be executed with '-u' command-line flag
        # and also the file object given with non-buffering mode (see constructor).
        return self

    def discard(self, msg):
        self.ostream.write("%s\n" % msg.replace("\n", ' '))
        # No need to flush here because the interpreter should be executed with '-u' command-line flag
        # and also the file object given with non-buffering mode (see constructor).
        return self

    def read(self):
        fd = self.istream.fileno()
        unpacker = msgpack.Unpacker()
        while fd in select.select([fd], [], [])[0]:
            data = os.read(fd, 2048)
            if not data:
                return

            unpacker.feed(data)
            for obj in unpacker:
                yield obj


def _translate(s, ttable):
    """
    Characters transliteration function to be used internally by `fixKeys` function.
    :param s: String to be processed.
    :param ttable: Transliteration table.
    :return: Transliterated string.
    """
    for fc, tc in ttable:
        s = s.replace(fc, tc)
    return s


def fixKeys(struct, frm=u'$.', to=u'\uff04\uff0e', ttable=None):
    """
    Fix key names to be valid for storing in MongoDB - it does not accept keys with '$.' symbols.
    Those symbols will be replaced with appropriate `to` symbols.

    :param struct: Data to be examined.
    :param frm: Characters to be translated.
    :param to: Characters to be placed in place of translating.
    :param ttable: Translation table. If no passed, will be constructed with `frm` and `to` arguments.
    :return: The same or updated structure.
    """

    if not ttable:
        ttable = zip(frm, to)

    if isinstance(struct, (list, tuple, set, )):
        for v in struct:
            fixKeys(v, ttable=ttable)

    if isinstance(struct, dict):
        updates = []
        for k, v in struct.iteritems():
            nk = _translate(k, ttable)
            if k != nk:
                updates.append((k, nk, ))
            if isinstance(v, (list, tuple, set, dict, )):
                fixKeys(v, ttable=ttable)

        for k, nk in updates:
            struct[nk] = struct.pop(k)

    return struct
