#!/skynet/python/bin/python

"""
The plugin suitable to process both V1 and V3 shard state reports.
"""

import pymongo
import datetime

import pkg_resources

pkg_resources.require('pymongo')
pkg_resources.require('msgpack-python')

import bson
import msgpack

pkg_resources.require('skynet-heartbeat-server-service')
from ya.skynet.services.heartbeatserver.bulldozer import helper


class Type(type):
    def __new__(mcs, value):
        if value == 'undef':
            return None
        return value


class OkType(Type):
    def __new__(mcs, value):
        value = Type(value)
        if value is None:
            return
        if value == 'OK' or value == True:
            return True
        elif value == False:
            return False
        else:
            return TextType(value)


class Md5Type(Type):
    def __new__(mcs, value):
        value = Type(value)
        if value is None:
            return

        if len(value) == 32:
            try:
                return value.decode('hex')
            except:
                return None


class ChkSumType(Md5Type):
    def __new__(mcs, value):
        if value.startswith('MD5:'):
            value = value[4:]
        return Md5Type.__new__(Md5Type, value)


class IntType(Type):
    def __new__(mcs, value):
        value = Type(value)
        if value is None:
            return
        try:
            return int(value)
        except:
            return


class TextType(Type):
    def __new__(mcs, value):
        value = Type(value)
        if value is None:
            return ''
        try:
            return str(value)
        except:
            return ''


STATE_FIELDS = {
    'created': ('\x00', IntType),
    'defer': ('\x01', IntType),
    'acquire': ('\x02', OkType),
    'sky-result': ('\x03', OkType),
    'skytries': ('\x04', IntType),
    'install': ('\x05', OkType),
    'generated_chksum': ('\x06', Md5Type),
    'handmade': ('\x07', OkType),
    'defer-waitlist': ('\x08', TextType),
    'remove': ('\x09', TextType),
    'conf_mtime': ('\x0a', IntType),
    'chksum': ('\xff', ChkSumType),
}


def compactState(state, hint=None):
    newState = {}
    for key, value in state.items():
        if key in STATE_FIELDS:
            idx, typ = STATE_FIELDS[key]
            v = value
            value = typ(value)
            if value is not None and value != '':
                newState[idx] = value
        else:
            newState[key] = value

    return newState


def main():
    dbcs = {}
    rdb = helper.ReportDatabase()
    com = helper.Communicator().ready()
    for host, name, data in com.read():
        try:
            dbc = dbcs[name] if name in dbcs else dbcs.setdefault(name, helper.ReportCollection(name, rdb=rdb))
            report = helper.fixKeys(data['report'])

            errors = set()
            version = report.pop('version', report.pop('v', None))
            if not version or version not in (1, 3):
                errors.add('Got unsupported ShardState report version: %r' % version)
            else:
                if 'error' in report:
                    errors.add(report['error'])

            state = {}
            if not errors:
                fullstate = report.get('s' if version == 3 else 'shards', {})

                for name, shardstate in fullstate.items():
                    for warn in shardstate.get('warnings', []):
                        errors.add('WARNING: %s: %s' % (name, warn, ))
                    if 'error' in shardstate:
                        errors.add('ERROR: %s: %s' % (name, shardstate['error']))
                    else:
                        s = compactState(shardstate.get('state', {})) if version < 3 else shardstate
                        if s:
                            state[name] = s

            update = {
                '$set': {
                    'state': bson.binary.Binary(msgpack.dumps(state)),
                    'version': version,
                    'last_update': datetime.datetime.now(),
                }
            }

            if errors:
                update['$set']['error'] = '\n'.join(errors)
            else:
                update['$unset'] = {'error': 1}
            dbc.coll.update({'host': host}, update, upsert=True)
        except (TypeError, ValueError) as ex:
            com.discard(repr(ex))
        else:
            com.ready()
