#!/skynet/python/bin/python

import re
import datetime
import itertools

import pkg_resources

pkg_resources.require('skynet-heartbeat-server-service')
from ya.skynet.services.heartbeatserver.bulldozer import helper


try:
    # For testing purposes only
    # from api.logger import constructLogger
    # log = constructLogger(source='', app='skyinfo', filename='skyinfo.log')
    log = None
except:
    log = None


def svnURL2Version(url):
    r = re.compile(
        r"""
            (?:svn\+ssh|https)://(?:\w+@)?arcadia.yandex.ru/arc/(?: # common prefix
                (trunk)|                                            # trunk variant
                tags/skynet/(?:                                     # tag variant
                    (?:(?:stable|testing)-([^/]+))|                 #   - old-styled
                    ([^/]+)                                         #   - modern
                )/|
                branches/skynet/([^/]+)/                            # branch variant
            )(?:[^@]+@(\d+))?                                       # optional revision number
        """,
        re.VERBOSE
    )
    m = r.match(url)
    groups = m.groups() if m else (url, None)
    if groups[1] is not None:  # Special case for old-styled versions
        return groups[1].replace('-', '.')
    return itertools.chain(
        itertools.ifilter(lambda x: x is not None, groups),
        itertools.repeat('UNKNOWN')
    ).next() + ('@%s' % groups[-1] if groups[-1] is not None else '')


def main():
    dbc = helper.ReportCollection('hostinfo')
    # lacmus = helper.LacmusReporter(log=log)

    com = helper.Communicator().ready()
    for host, type, data in com.read():
        try:
            r = data['report']
            remove = None
            update = None
            now = datetime.datetime.now()

            if type == 'svninfo':
                update = {
                    'skynet.version': svnURL2Version(r['svn_url']),
                    'skynet.updated': now,
                    'skynet.svn': {'url': r['svn_url'], 'revision': r['svn_revision']},
                }

            elif type == 'skyinfo':
                url = r.get('url', r.get('_url', ''))
                update = {
                    'skynet.version': r['version'],
                    'skynet.updated': now,
                    'skynet.svn': {'url': url, 'revision': r['revision']},
                    'skynet.installed': datetime.datetime.fromtimestamp(r['timestamp']),
                    'skynet.srvmngr': helper.fixKeys(r['srvmngr']),
                }
                old_skycore_report = r.get('skycore', None)
                if old_skycore_report is not None:
                    update['skynet.skycore'] = helper.fixKeys(old_skycore_report)

                if 'porto' in r:
                    update['porto.client'] = r['porto'].get('client', None)
                    update['porto.server'] = r['porto'].get('server', None)

            elif type == 'goskyversion':
                ver = r.get('version', None)
                if ver is None:
                    update = {'gosky.error': r['error']}
                    remove = {'gosky.version': ''}
                else:
                    update = {'gosky.version': ver}
                    remove = {'gosky.error': ''}

            elif type == 'skyinstall':
                update = {
                    'gosky.exitcode': r['exitcode'],
                    'skynet.version': r['version'],
                    'skynet.updated': now
                }

                if r['installed'] is not None:
                    update['skynet.installed'] = datetime.datetime.fromtimestamp(r['installed'])

                if r['exitcode'] != 0:
                    update['skynet.stderr'] = r.get('stderr')
                    update['skynet.reason'] = r.get('reason')
                else:
                    remove = {
                        'skynet.stderr': '',
                        'skynet.reason': '',
                    }

                if r['exitcode'] == 12:
                    remove = {
                        'skynet.srvmngr': ''
                    }

                if r['svn']['url'] is not None:
                    update['skynet.svn'] = {'url': r['svn']['url'], 'revision': r['svn']['revision']}

            elif type == 'skycoreupdate':
                update = {
                    'skynet.updated': now,
                    'skynet.skycore_updated': now
                }
                if 'name' in r:
                    # old format
                    update['skynet.skycore.services.' + r['name']] = {
                        'state': r['state'],
                        'version': r['version']
                    }
                elif 'services' in r:
                    # FIXME: workaround for bug in skycore < 16.10
                    for service, state in r['services'].iteritems():
                        update['skynet.skycore.services.' + service] = {
                            'state': state['state'],
                            'version': state['version']}
                else:
                    # new format
                    for service, state in r.iteritems():
                        update['skynet.skycore.services.' + service] = {
                            'state': state['state'],
                            'version': state['version']}

            elif type == 'skycoreinfo':
                update = {
                    'skynet.updated': now,
                    'skynet.skycore_updated': now,
                    'skynet.skycore': helper.fixKeys(r),
                }

            if update:
                dbc.update(host, update, remove)

                # DON'T send reports to lacmus v2
                # signals = {
                #     'skynet.version': update.get('skynet.version', None),
                #     'gosky.version': update.get('gosky.version', None),
                #     'skynet.deploy_type': update.get('skynet.srvmngr', {}).get('deploy_type', None),
                #     'porto.client': update.get('porto.client', None),
                #     'porto.server': update.get('porto.server', None),
                # }
                # for sig_name, sig_value in signals.items():
                #     lacmus.add_signal(sig_name, sig_value)
                #
                # lacmus.send_report(host, data.get('end', time.time()))

                com.ready()
            else:
                com.discard('Unknown report type %r' % type)
        except (KeyError, TypeError, ValueError, AttributeError) as ex:
            com.discard(repr(ex))
