#!/skynet/python/bin/python

import os
import logging
from sending import parse_hosts_from_url, InstanceUsageBufferingSender
from common import InstanceTypes

logging.basicConfig(level=logging.DEBUG)


def is_bad_report(report):
    return any(v is None for v in report.itervalues())


def report_to_values(template, host, report):
    try:
        values = template.format(host=host, **report)
        return "({})".format(values)
    except KeyError:
        return None


def insert_query(values_template, query_template, hosts_and_reports):
    values = [report_to_values(values_template, host, report) for host, report in hosts_and_reports]
    good_values = filter(None, values)
    if len(values) != len(good_values):
        logging.warn('dropping %i reports', len(values) - len(good_values))
    if not good_values:
        return None
    return query_template.format(values=", ".join(good_values))


def insert_into_usage_query(report_type, hosts_and_reports):
    values_templates = {
        InstanceTypes.ISS: (
            "'{host}', {port}, {ts}, '{group}', '{formatted_date}', "
            "{major_tag}, {minor_tag}, {version}, "
            "{instance_mem_usage}, {instance_cpu_usage}, {instance_anon_mem_usage}, {instance_net_rx_usage}, {instance_net_tx_usage}"
        ),
        InstanceTypes.HOST: (
            "'{host}', {ts}, '{formatted_date}', "
            "{instance_mem_usage}, {instance_cpu_usage}, "
            "{hdd_total}, {hdd_usage}, "
            "{ssd_total}, {ssd_usage}"
        ),
        InstanceTypes.QLOUD: (
            "'{host}', '{port}', {ts}, '{formatted_date}', "
            "'{service}', '{environ}', '{app}', '{project}', "
            "'{install_type}', {instance_mem_usage}, {instance_cpu_usage}, {instance_net_rx_usage}, {instance_net_tx_usage}"
        ),
        InstanceTypes.OPENSTACK: (
            "'{host}', '{port}', {ts}, '{formatted_date}', "
            "'{tenant_id}', {instance_mem_usage}, {instance_cpu_usage}"
        ),
        InstanceTypes.YP: (
            "'{host}', '{port}', {ts}, '{group}', '{formatted_date}', "
            "{instance_mem_usage}, {instance_mem_guarantee}, {instance_cpu_usage}, {instance_cpu_guarantee}, {instance_anon_mem_usage}, {instance_net_rx_usage}, {instance_net_tx_usage}"
        ),
    }
    tables = {
        InstanceTypes.ISS: """
            instanceusage (
                host, port, ts, group, eventDate,
                major_tag, minor_tag, version,
                instance_memusage, instance_cpuusage, instance_anon_memusage,
                instance_net_rx, instance_net_tx
            )
        """,
        InstanceTypes.HOST: """
            hostusage (
                host, ts, eventDate, mem_usage, cpu_usage,
                hdd_total, hdd_usage, ssd_total, ssd_usage
            )
        """,
        InstanceTypes.QLOUD: """
            qloudusage (
                host, instanceId, ts, eventDate,
                service, environ, app, project, installType,
                mem_usage, cpu_usage, net_rx, net_tx
            )
        """,
        InstanceTypes.OPENSTACK: """
            openstackusage (
                host, instanceId, ts, eventDate, projectId,
                mem_usage, cpu_usage
            )
        """,
        InstanceTypes.YP: """
            ypusage(
                host, port, ts, group, eventDate,
                instance_memusage, instance_memallocated, instance_cpuusage, instance_cpuallocated, instance_anon_memusage,
                instance_net_rx, instance_net_tx
            )
        """,
    }

    query_template = "INSERT INTO {table} VALUES {{values}};".format(table=tables[report_type])

    return insert_query(values_templates[report_type], query_template, hosts_and_reports)


def generate_clickhouse_query(report_type, hosts_and_reports):
    if report_type not in InstanceTypes.ALL:
        logging.error("Unknown report type %s", report_type)
    else:
        good_hosts_and_reports = [(host, report) for host, report in hosts_and_reports if not is_bad_report(report)]

        if len(good_hosts_and_reports) != len(hosts_and_reports):
            logging.error("Dropping %i reports with Nones!", len(hosts_and_reports) - len(good_hosts_and_reports))

        return [insert_into_usage_query(report_type, good_hosts_and_reports)]


def main():
    chs = InstanceUsageBufferingSender(
        parse_hosts_from_url(os.environ['DATABASE_URI']),  # e. g. DATABASE_URI=clickhouse://man1-8406.search.yandex.net:17352,vla1-4552.search.yandex.net:17352
        generate_clickhouse_query,
        avg_seconds_between_queries=1.5,
    )
    chs.run()
