from __future__ import absolute_import, print_function, division

import re
import yaml


REP_RE = re.compile('\$\{([a-zA-Z0-9_\-\.]+)\}')
RE_PLACEHOLDER = re.compile(r'\${([a-z0-9_.-]+)}', re.IGNORECASE)


class AppConfigSubConfigProxy(object):
    def __init__(self, config, key):
        self.__config = config
        self.__key = key

    def __iter__(self):
        return self.__config.iter(self.__key)

    def __getattr__(self, attr):
        return self.__config.__getattr__('.'.join((self.__key, attr)))


class AppConfig(dict):
    __proxyClass = AppConfigSubConfigProxy

    def __init__(self):
        self.__needProxy = {}

    def _flattenData(self, data):
        flattenData = {}
        left = [(None, k, v) for k, v in data.items()]

        while left:
            prefix, key, value = left.pop(0)

            if prefix is not None:
                key = '.'.join((prefix, key))

            if isinstance(value, dict):
                if not value.pop('_dict', 0):
                    for subKey, subValue in value.items():
                        left.append((key, subKey, subValue))
                    continue

            flattenData[key] = value

        return flattenData

    def load(self, data):
        self.update(self._flattenData(data))

        toParse = self.keys()

        while toParse:
            key = toParse.pop(0)
            value = self[key]

            if '.' in key:
                tail = None
                for sKey in key.split('.')[:-1]:
                    if tail is not None:
                        sKey = '.'.join((tail, sKey))
                    self.__needProxy.setdefault(sKey, None)
                    tail = sKey

            if not isinstance(value, basestring) or '$' not in value:
                continue

            while True:
                match = REP_RE.search(value)
                if not match:
                    break

                replKey = match.group(1)

                # If replacement didnt parsed yet, continue
                if replKey in toParse:
                    toParse.append(key)
                    break

                start, end = match.span()

                if start == 0 and end == len(value) and not isinstance(self[replKey], basestring):
                    value = self[replKey]
                    break
                else:
                    value = ''.join((value[0:start], self[replKey], value[end:]))

            self[key] = value

    def iter(self, key):
        key += '.'
        for i in filter(lambda x: x.startswith(key), self.keys()):
            yield i[len(key):]

    def __getattr__(self, attr):
        if attr in self.__needProxy:
            if self.__needProxy[attr] is None:
                self.__needProxy[attr] = self.__proxyClass(self, attr)
            return self.__needProxy[attr]

        if attr in self:
            return self[attr]
        raise AttributeError('No such config attribute %r' % attr)


class SlottedDict(dict):
    def __getattr__(self, key):
        value = self[key]
        if isinstance(value, dict):
            value = self[key] = type(self)(value)
        self.__dict__[key] = value
        return value

    def __setattr__(self, key, value):
        self[key] = self.__dict__[key] = value


def _dict_query(dct, path):
    for k in path.split('.') if path else []:
        dct = dct[k]
    return dct


def _config_evaluate(data):
    # Recursively evaluate placeholders inplace
    queue = [data]

    while queue:
        next_queue = []
        for item in queue:
            if isinstance(item, dict):
                items = item.items()
            elif isinstance(item, list):
                items = enumerate(item)

            for k, v in items:
                if isinstance(v, (dict, list)):
                    next_queue.append(v)
                elif isinstance(v, basestring):
                    changed = False
                    while True:
                        match = RE_PLACEHOLDER.search(v)
                        if not match:
                            break
                        v = v[:match.start()] + _dict_query(data, match.group(1)) + v[match.end():]
                        changed = True
                    if changed:
                        item[k] = v

            queue = next_queue


def loadConfig(path, overrides={}):
    data = yaml.load(open(path, 'rb'))
    config = (
        data['subsections']['skynet']['subsections']['services']['subsections']['heartbeat-server']
        ['config']['config']
    )
    config.update(overrides)
    _config_evaluate(config)

    return SlottedDict(config)
