from __future__ import absolute_import, print_function, division

import daemon
import os
import pprint
import faulthandler
import signal
import sys

from . import config, logger


def magicPrint(*args, **kwargs):
    sep = kwargs.pop('sep', ' ')
    end = kwargs.pop('end', '\n')
    stream = kwargs.pop('file', sys.stdout)

    if len(args) != 1 or isinstance(args[0], basestring) and '\n' in args[0]:
        return oprint(*args, sep=sep, end=end, file=stream)

    return pprint.pprint(args[0], stream=stream)


class Context(dict):
    def __init__(self):
        # Initialize all variables
        self.cfg = None
        self.log = None
        self.__initialized = False

        super(Context, self).__init__()

    def initialize(self, args):
        assert not self.__initialized, 'Context already initialized'

        # Read app config
        self.cfg = config.loadConfig(args.config, overrides=dict(x.split('=', 1) for x in args.C))
        self.initializeLogging(args.verbose)

        # Hack print()
        if 'oprint' not in __builtins__:
            __builtins__['oprint'] = __builtins__['print']
            __builtins__['print'] = magicPrint

        if args.trace_config:
            pprint.pprint(self.cfg, stream=sys.stderr)
            raise SystemExit(0)

        # Create workdir
        if not os.path.exists(self.cfg.WorkdirPath):
            os.makedirs(self.cfg.WorkdirPath)

        self.dctx = daemon.DaemonContext(  # pylint: disable=E1101
            #working_directory=self.cfg.WorkdirPath,
            umask=022,
            stdout=None if args.daemonize else sys.stdout,
            stderr=None if args.daemonize else sys.stderr,
            stdin=None if args.daemonize else sys.stdin,
            prevent_core=False,
            detach_process=args.background,
            signal_map=self.initializeSignalHandlers(),
        )
        daemon.daemon.set_signal_handlers = lambda x: None

        self.__isOpen = False
        self.__initialized = True

        return self

    def setSignalHandlers(self, signalHandlerMap):
        try:
            import gevent.signal as gevent_signal
        except ImportError:
            import gevent as gevent_signal

        for signum, handler in signalHandlerMap.items():
            if handler == 1:
                handler_fun = lambda signum=signum, frame=None: None
            else:
                handler_fun = lambda signum=signum, frame=None: handler(signum, frame)
            gevent_signal.signal(signum, handler_fun)

    def setFaultHandler(self):
        faulthandler.enable()

    def initializeSignalHandlers(self):
        signalMap = {}

        for signame in 'SIGTTIN', 'SIGTTOU', 'SIGPIPE':
            if hasattr(signal, signame):
                signalMap[getattr(signal, signame)] = None

        for signame in 'SIGTERM', 'SIGALRM', 'SIGHUP', 'SIGTSTPx':
            if hasattr(signal, signame):
                signalMap[getattr(signal, signame)] = self.terminate

        return signalMap

    def initializeLogging(self, verbosity):
        # Initialize logging
        logger.setupLogging(self)              # Initialize logging from config

        # Capture all warnings. We know this function exists!
        logger.logging.captureWarnings(True)  # pylint: disable=E1101

        self.log = logger.logging.getLogger()         # Make root logger
        self.log.name = 'main'                 # Change root logger name

        if verbosity > 0:
            for handler in self.log.handlers:
                if hasattr(handler, 'stream') and handler.stream in (sys.stdout, sys.stderr):
                    handler.level -= verbosity * 10

    def terminate(self, signum, frame=None):
        signame = None
        try:
            for key, value in signal.__dict__.items():
                if value == signum:
                    signame = key
                    break
        except Exception as ex:
            signame = 'unknown: {0}'.format(ex)

        raise SystemExit('Terminating on signal {0} ({1})'.format(signum, signame))

    def __getattr__(self, attr):
        return self[attr]

    def __setattr__(self, attr, value):
        self[attr] = value

    def __enter__(self):
        assert not self.__isOpen
        self.dctx.open()
        self.setSignalHandlers(self.dctx._make_signal_handler_map())
        self.setFaultHandler()
        self.__isOpen = True
        return self

    def __exit__(self, excType, excValue, traceback):
        assert self.__isOpen
        self.dctx.close()
        self.__isOpen = False

    def __repr__(self):
        return '<Context>'
