"""
The purpose of this module is to make all needed stuff for starting daemon.
This includes initialization of our Context with DaemonContext inside.

During context initialization we will close file descriptors, setup
signal handlers and so on.

It is important what before context initialization gevent will NOT be imported.
"""

from __future__ import absolute_import, print_function, division

import argparse
import os
import sys

from kernel.util.errors import formatException

from .context import Context


def parseArgs():
    parser = argparse.ArgumentParser(
        formatter_class=(
            lambda *args, **kwargs:
            argparse.ArgumentDefaultsHelpFormatter(*args, width=120, **kwargs)
        ),
        description='Run copierng daemon.'
    )
    parser.add_argument('-s', '--state', action='store_true', help=argparse.SUPPRESS)
    parser.add_argument('-d', '--daemonize', action='store_true')
    parser.add_argument('-b', '--background', action='store_true')
    parser.add_argument(
        '-v', '--verbose', metavar='N', nargs='?', const=1, type=int, default=0,
        help='Increase verbosity by N times.'
    )
    parser.add_argument(
        '-C', action='append', metavar='KEY=VALUE', default=[],
        help='Override config (can be specified several times)'
    )
    parser.add_argument(
        '-c', '--config', required=True,
        help='Config file'
    )
    parser.add_argument('--trace-config', action='store_true', help='Trace configuration to stdout')

    return parser.parse_args()


def state():
    # backward compat
    print('PASSIVE-CHECK:heartbeat-daemon;2;Unable to connect to the service: why do you call this')
    return 0


def main():
    # Initialize daemon context
    try:
        args = parseArgs()
        if args.state:  # KORUM: Otherwise, for some reason, each print() output will be quoted..
            return state()

        ctx = Context()
        ctx.initialize(args)
    except SystemExit:
        raise
    except BaseException:
        print(
            'ERROR (initializing): {0}'.format(formatException()),
            file=sys.stderr
        )
        os._exit(1)  # pylint: disable=W0212

    try:
        with ctx:
            from ..main import main as realMain
            return realMain(ctx)
    except SystemExit:
        raise
    except BaseException:
        ctx.log.critical('Unhandled error: %s' % (formatException(), ))
        os._exit(1)  # pylint: disable=W0212


if __name__ == '__main__':
    raise SystemExit(main())
