# vim: set foldmethod=marker

from __future__ import absolute_import, print_function, division

import os
import time
import random
import traceback as tb

import gevent
import gevent.event
import gevent.queue

from kernel.util.console import setProcTitle
from kernel.util.errors import formatException

from . import restapi
from .server import Server
from .rpc.server import RPC
from .bulldozer import Brigadier


class HeartBeatServer(object):
    # Initialization {{{
    def __init__(self, ctx):
        self.ctx = ctx
        self.log = ctx.log.getChild('heartbeat-server')
        self.log.info('Initializing')

        self._stopFlag = gevent.event.Event()
        self._startFlag = gevent.event.Event()

        self._rpc = None
        self._server = None
        self._restapi = None
        self._brigadier = None
    # Initialization }}}

    # Management (start, stop, join) {{{
    def start(self):
        assert not self._stopFlag.isSet()

        self._server = Server(self.ctx).start()
        self._brigadier = Brigadier(self.ctx).start()
        self._rpc = RPC(self.ctx).start()

        # Register server connection handlers
        self._server.registerConnectionHandler(self._rpc.getConnectionHandler())

        # Register RPC callbacks
        self._rpc.registerHandler('ping', self._ping)
        self._rpc.registerHandler('stop', self._stop)
        self._rpc.registerHandler('status', self._status)
        self._rpc.registerHandler('report', self._report)
        self._rpc.registerHandler('timesync', self._timesync)

        self._restapi = restapi.Server(self.ctx, self._brigadier).start()

        self._stopFlag.clear()
        self._startFlag.set()
        return self

    def stop(self):
        assert self._startFlag.isSet()

        self._restapi.stop().join()
        self._brigadier.stop()
        self._server.stop().join()
        self._rpc.stop().join()
        self._stopFlag.set()
        self._startFlag.clear()
        return self

    def join(self):
        self._stopFlag.wait()
    # Management (start, stop, join) }}}

    # RPC Meths {{{
    @RPC.full
    def _ping(job, self):
        if self._startFlag.wait(timeout=30):
            # Check also proxy status (actually, bulldozers manager status).
            return self._brigadier.ping()
        job.log.warning('Ping timed out waiting startFlag to become set (wait 30 seconds)')
        return False

    @RPC.simple
    def _stop(self):
        self._stopFlag.set()

    @RPC.simple
    def _status(self):
        return self._brigadier.status()

    @RPC.full
    def _report(job, self, mode, src, sent, reports):
        errors = []
        now = time.time()
        queue = gevent.queue.Queue()

        # First of all, update host's last contact time.
        self._brigadier.on_contact(src)

        # Now check the time difference.
        # If packet time difference more than 10 seconds -- require time sync
        # We allow 10 seconds skew if sync was not yet performed (mode == '2')
        # and 80 seconds (60 sec req timeout + 20 sec max skew) if it was (mode == '2S')
        if abs(sent - now) > (80 if mode == '2S' else 10):
            if mode == '2':
                job.log.debug(
                    'Ask host %s to sync with us (time difference is %0.6f seconds)' %
                    (src, sent - now)
                )
            else:
                job.log.warning(
                    'Host %s failed to sync time with us (time difference is %0.6f seconds)' %
                    (src, sent - now)
                )
            return 'TIMESYNC'

        for idx, name, report in reports:
            try:
                valid = report.get('valid', None)
                expires = float('inf') if valid is None else now + valid + report['end'] - sent
                self._brigadier.process(
                    src, report, expires,
                    lambda x, idx=idx, name=name: queue.put((idx, name, x))
                )
            except Exception as ex:
                msg = 'Error schedule report #%d from %r of type %r for processing: %%s' % (idx, src, name)
                queue.put((idx, name, msg % str(ex), ))
                job.log.error(msg % tb.format_exc())

        for _ in range(len(reports)):
            idx, name, data = queue.get()
            if not data:
                job.state((idx, name, ))
            else:
                self.log.debug('Host %r error processing report %r: %s', src, name, data)
                errors.append(data)

        return 'OK' if not errors else errors

    @RPC.simple
    def _timesync(self):
        return time.time()
    # RPC Meths }}}


def main(ctx):
    log = ctx.log

    setProcTitle(ctx.cfg.ProgramName)
    log.info('Initializing {0}'.format(ctx.cfg.ProgramName))

    random.seed()

    app = HeartBeatServer(ctx)
    app.start()

    try:
        app.join()
    except KeyboardInterrupt:
        log.info('Caught SIGINT, stopping daemon')
    except SystemExit as ex:
        if ex.args:
            # We expect exception if this case will have meaningfull message
            log.warning(ex)
        else:
            log.warning('Got SystemExit exception in main loop')
        raise
    except BaseException:
        log.critical('Unhandled exception: %s, exit immidiately!' % formatException())
        os._exit(1)  # pylint: disable=W0212

    app.stop()

    with gevent.Timeout(2) as timeout:
        try:
            app.join()
        except gevent.Timeout as err:
            if err != timeout:
                raise
            log.error('Failed to stop {0!r} gracefully -- timeout occured.'.format(app))
            os._exit(1)  # pylint: disable=W0212

    log.info('Stopped {0}'.format(ctx.cfg.ProgramName))
    os._exit(0)
    return 0
