from __future__ import absolute_import

import time
import socket
import httplib

import flask
from flask.ext import restful
from flask.ext.restful import reqparse

import gevent
import gevent.pywsgi
import gevent.socket

from . import bulldozer


# URL base path, which should be prefixes eny API URL.
BASE_URL_PATH = '/api/v1.0'


class Report(restful.Resource):
    def __init__(self):
        self.reqparse = reqparse.RequestParser()
        self.reqparse.add_argument(
            'name',
            type=str, required=True, help='Report type (name)', location='json'
        )
        self.reqparse.add_argument(
            'source',
            type=str, help='Report source (autodetect if none)', location='json'
        )
        self.reqparse.add_argument(
            'report',
            type=lambda x: x, required=True, help="Report content (dictionary or compressed string)", location='json'
        )
        self.reqparse.add_argument(
            'compression',
            type=str, help="Report content compression method (can be 'msgpack', 'zip' or 'bz2')", location='json'
        )
        self.reqparse.add_argument(
            'valid',
            type=int, help="Amount of seconds the report will be valid", default=3600, location='json'
        )
        super(Report, self).__init__()

    def post(self):
        error = lambda code, message: ({'status': code, 'message': message}, code)
        args = self.reqparse.parse_args()
        srv = flask.g.restsrv
        src = args.get('source')
        name = args.get('name')
        if not src:
            src = flask.request.headers.getlist("X-Forwarded-For")
            src = gevent.socket.getfqdn(src[0] if src else flask.request.remote_addr)
            if not src:
                return error(httplib.UNPROCESSABLE_ENTITY, 'Unable to determine source address')

        valid = args.get('valid')
        data = args.get('report')
        comp = args.get('compression')
        expires = float('inf') if valid is None else time.time() + valid
        report = {'name': name, 'report': data, 'compression': comp}
        try:
            ret = srv.brigadier.process(src, report, expires)
            if ret:
                srv.log.warn("Report of type %r from %r not processed: %r", name, src, ret)
                return error(httplib.UNPROCESSABLE_ENTITY, ret)
            else:
                srv.log.debug("Report of type %r from %r processed successfully.", name, src)
                return '', httplib.CREATED
        except bulldozer.UnsupportedReport as ex:
            srv.log.warn(str(ex))
            return error(httplib.UNPROCESSABLE_ENTITY, str(ex))


class Server(object):
    def __init__(self, ctx, brigadier):
        self.ctx = ctx
        self.cfg = ctx.cfg.rest
        self.log = ctx.log.getChild(self.__module__.rsplit('.', 1)[-1]).getChild(self.__class__.__name__)
        self.brigadier = brigadier
        self.hostname = socket.gethostname()
        self.log.debug('Initializing')

        self._api = None
        self._wsgi = None
        self._flask = None

    def _setRequestContext(self):
        flask.g.restsrv = self

    def _setCommonHeaders(self, resp):
        resp.headers['X-Heartbeat-Server'] = self.hostname
        return resp

    def start(self):
        self._flask = flask.Flask(__name__)
        self._flask.config.update(self.cfg.flask_config)
        self._flask.before_request(self._setRequestContext)
        self._flask.after_request(self._setCommonHeaders)
        self._api = restful.Api(self._flask)

        self._api.add_resource(Report, BASE_URL_PATH + '/report')

        self._sock = gevent.socket.socket(gevent.socket.AF_INET6)
        self._sock.setsockopt(gevent.socket.SOL_SOCKET, gevent.socket.SO_REUSEADDR, 1)
        self._sock.setsockopt(gevent.socket.IPPROTO_IPV6, gevent.socket.IPV6_V6ONLY, 0)
        self._sock.bind((self.cfg.host, self.cfg.port))
        self._sock.listen(self.cfg.backlog)

        self._wsgi = gevent.pywsgi.WSGIServer(
            self._sock,
            application=self._flask,
            log=None
        )
        self._wsgi.start()

        self.log.info('Listening for HTTP at %r:%d', self.cfg.host, self.cfg.port)
        return self

    def stop(self):
        self._wsgi.stop(timeout=self.cfg.stop_timeout)
        self._sock.close()
        self.log.info('Stopped HTTP server')
        return self

    def join(self):
        self._wsgi._stopped_event.wait()
        return self

    def report(self, name):
        pass
