import gevent
import gevent.socket
import socket
import struct
import os

from kernel.util.errors import formatException

from .utils.socket import Socket, EOF


class ConnectionHandler(object):
    def __init__(self):
        self.__magicPacker = struct.Struct('!I')

    def packMagic(self, magicInteger):
        assert 0 <= magicInteger < 2 ** 32
        return self.__magicPacker.pack(magicInteger)

    def getMagic(self):
        raise NotImplementedError

    def handleConnection(self, conn, addr):
        raise NotImplementedError


class Server(object):
    def __init__(self, ctx):
        self.ctx = ctx
        self.cfg = ctx.cfg.server

        self.log = ctx.log.getChild('server')
        self.log.debug('Initializing')

        self.__connectionHandlers = {
            'CHK ': lambda sock: sock.close() or self.log.debug('Got CHK packet, closing connection')
        }
        self.__workerGrn = None

    def registerConnectionHandler(self, handler):
        assert isinstance(handler, ConnectionHandler)

        magic = handler.getMagic()
        assert isinstance(magic, basestring) and len(magic) == 4

        self.__connectionHandlers[magic] = handler.handleConnection

    def start(self):
        assert self.__workerGrn is None
        log = self.log.getChild('start')

        unix = self.cfg.unix
        if unix:
            unixLock = self.cfg.unixLock
        else:
            unixLock = None

        host = self.cfg.host
        port = int(self.cfg.port) if self.cfg.port is not None else None

        backlog = int(self.cfg.backlog)

        if unix:
            assert unix and unixLock
            assert unix and port is None and host is None

        family = socket.AF_INET6 if not unix else socket.AF_UNIX

        self.__sock = gevent.socket.socket(family, socket.SOCK_STREAM)
        self.__sock.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)

        if not unix:
            self.__sock.setsockopt(socket.IPPROTO_IPV6, socket.IPV6_V6ONLY, 0)
            self.__sock.bind((host, port))
        else:
            sockDir = os.path.dirname(unix)
            if not os.path.exists(sockDir):
                os.makedirs(sockDir)

            if os.path.exists(unix):
                sock2 = gevent.socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)

                # We can wait a lot if other process will not accept connection
                # (e.g. if it is in STSTP state).
                sock2.settimeout(10)
                try:
                    sock2.connect(unix)
                    sock2.sendall('CHK ')
                except Exception as ex:
                    if isinstance(ex, socket.timeout):
                        raise
                else:
                    log.critical('We connected to socket \'%s\' -- probably another instance running' % (unix, ))
                    raise SystemExit(2)
                finally:
                    sock2.close()

                os.unlink(unix)

            self.__sock.bind(unix)
            try:
                oldUmask = os.umask(022)
                os.chmod(unix, 0666)
            finally:
                os.umask(oldUmask)

        self.__sock.listen(backlog)

        if not unix:
            log.info('Listening on: *:%d (backlog: %d)', port, backlog)
        else:
            log.info('Listening on: unix:%s (backlog: %d)', unix, backlog)

        self.__workerGrn = gevent.spawn(self.__workerLoop)
        return self

    def stop(self):
        if self.__workerGrn is not None:
            self.__workerGrn.kill(gevent.GreenletExit)
            try:
                self.__sock.shutdown(socket.SHUT_RDWR)
            except:
                pass
            self.__sock.close()
            try:
                unix = self.cfg.unix
                os.unlink(unix)
                self.__unixLockFp.close()
            except:
                pass
        return self

    def join(self):
        self.__workerGrn.join()

    def __worker(self, conn, addr):
        log = self.log.getChild('worker')
        if addr != '':
            log.debug('New connection from %r', addr)
        else:
            log.debug('New connection (unix socket)')

        sock = Socket(conn, geventMode=True)

        try:
            magic = sock.read(4, timeout=self.cfg.magic_receive_timeout)
        except gevent.Timeout:
            self.log.warning('Magic receive timeout for client %r', sock.addr)
            sock.close()
            return
        except EOF:
            log.warning('Got EOF while receiving magic from %r', sock.addr)
            sock.close()
            return

        if magic not in self.__connectionHandlers:
            log.warning('No handlers for magic %r for client %r', magic, sock.addr)
            sock.close()
            return

        try:
            self.__connectionHandlers[magic](sock)
        except Exception:
            log.warning('Got unhandled error while running connection handler %s: %s, client %r',
                self.__connectionHandlers[magic],
                formatException(),
                sock.addr
            )
            sock.close()

    def __workerLoop(self):
        log = self.log.getChild('worker')
        log.info('Started')

        while 1:
            try:
                conn, addr = self.__sock.accept()
                gevent.spawn(self.__worker, conn, addr)
            except gevent.GreenletExit:
                log.info('Received stop signal')
                break
            except Exception:
                log.error('Unhandled exception: %s', formatException())
                gevent.sleep(0.1)  # avoid busy loops
