import os
import sys
import threading
import atexit
import gevent.socket
import gevent.coros
from Queue import Queue, Empty as QueueEmpty


class DeblockProxy(object):
    def __init__(self, func, lock, put, get):
        self.__func = func
        self.__lock = lock
        self.__put = put
        self.__get = get

    def __call__(self, *args, **kwargs):
        with self.__lock:
            self.__put((self.__func, args, kwargs))
            return self.__get()


class Deblock(object):
    """
    Deblock obj.
    All variables and functions have "__" prefix here, because
    we dont want to callide with args/meths from object we extend.
    """

    def __init__(self, obj, keepalive=10):
        self.__obj = obj
        self.__rpipe = None
        self.__wpipe = None
        self.__queue = None
        self.__thread = None
        self.__result = None
        self.__lock = threading.Lock()            # generic lock for thread
        self.__glock = gevent.coros.Semaphore(1)  # this will prevent >1 waiter at a time
        self.__atexitRegistered = False
        self.__keepalive = keepalive

    def __runthread(self):
        if not self.__atexitRegistered:
            atexit.register(self.__stopthread)
            self.__atexitRegistered = True

        self.__rpipe, self.__wpipe = os.pipe()
        self.__queue = Queue()
        self.__thread = threading.Thread(target=self.__loop, name='Deblock %s' % (self.__obj, ))
        self.__thread.daemon = True
        self.__thread.start()

    def __stopthread(self):
        with self.__lock:
            if self.__thread:
                self.__queue.put(None)
                self.__thread.join(timeout=10)
                self.__thread = None

    def __loop(self):
        while True:
            try:
                func, args, kwargs = self.__queue.get(timeout=self.__keepalive)
            except QueueEmpty:
                break
            except TypeError:
                return

            try:
                self.__result = func(*args, **kwargs)
                os.write(self.__wpipe, 'y')
            except:
                self.__result = sys.exc_info()
                os.write(self.__wpipe, 'n')

        # Remove ourselves reference, to clean up resources
        with self.__lock:
            self.__thread = None
            self.__queue = None
            os.close(self.__rpipe)
            os.close(self.__wpipe)

    def __getresult(self):
        gevent.socket.wait_read(self.__rpipe)
        restype = os.read(self.__rpipe, 1)

        if restype == 'y':
            return self.__result
        elif restype == 'n':
            raise self.__result[0], self.__result[1], self.__result[2]
        else:
            raise RuntimeError('Unknown error occured (%r)!' % (restype, ))

    def __deblock(self, obj):
        with self.__lock:
            if self.__thread is None or not self.__thread.isAlive():
                self.__runthread()
        return DeblockProxy(obj, self.__glock, self.__queue.put, self.__getresult)

    def __getattr__(self, name):
        m = getattr(self.__obj, name)
        if callable(m):
            return self.__deblock(m)
        return m

    def __call__(self, *args, **kwargs):
        return self.__deblock(self.__obj)(*args, **kwargs)

    def __repr__(self):
        return '<Deblock %r>' % (self.__obj, )
