# vim: foldmethod=marker

from __future__ import absolute_import, print_function, division

import socket
try:
    import gevent
except ImportError:
    gevent = None
import msgpack
import errno
import time


class EOF(Exception):
    pass


class Socket(object):
    def __init__(self, sock, geventMode=False):
        self.sock = sock
        self.__geventMode = geventMode
        self.__closed = False

        sockname = sock.getsockname()
        if sockname == '':
            self.__addr = ('', '')  # socketpair?
            self.__binded = True
        elif sockname[1] == 0:
            self.__addr = None
            self.__binded = False
        else:
            self.__addr = sock.getsockname()
            self.__binded = True

        try:
            self.__peer = sock.getpeername()
            self.__connected = True
        except socket.error as ex:
            if ex.errno == errno.ENOTCONN:
                self.__peer = None
                self.__connected = False
            else:
                raise

    # Properties and shortcuts {{{
    @property
    def addr(self):
        assert self.__binded or self.__connected, 'Bind first'
        if self.__addr is None:
            self.__addr = self.sock.getsockname()
        return self.__addr

    @property
    def peer(self):
        assert self.__connected, 'Connect first'
        if self.__peer is None:
            self.__peer = self.sock.getpeername()

        return self.__peer

    @property
    def nodelay(self):
        return self.sock.getsockopt(socket.IPPROTO_TCP, socket.TCP_NODELAY)

    @nodelay.setter
    def nodelay(self, value):
        self.sock.setsockopt(socket.IPPROTO_TCP, socket.TCP_NODELAY, value)

    @property
    def sendBuffer(self):
        return self.sock.getsockopt(socket.SOL_SOCKET, socket.SO_SNDBUF)

    @sendBuffer.setter
    def sendBuffer(self, value):
        self.sock.setsockopt(socket.SOL_SOCKET, socket.SO_SNDBUF, value)

    @property
    def receiveBuffer(self):
        return self.sock.getsockopt(socket.SOL_SOCKET, socket.SO_RCVBUF)

    @receiveBuffer.setter
    def receiveBuffer(self, value):
        self.sock.setsockopt(socket.SOL_SOCKET, socket.SO_RCVBUF, value)

    @property
    def timeout(self):
        return self.sock.gettimeout()

    @timeout.setter
    def timeout(self, value):
        self.sock.settimeout(value)

    # Properties and shortcuts }}}

    def close(self, shutdown=socket.SHUT_RDWR):
        if self.__closed:
            return

        if shutdown:
            try:
                self.sock.shutdown(shutdown)
            except:
                pass

        self.sock.close()

        self.__closed = True

    def recv(self, readby=8192, timeout=None):
        if timeout:
            otimeout = self.timeout
            try:
                self.timeout = timeout
                return self.sock.recv(readby)
            except socket.error as ex:
                if str(ex) == 'timed out':
                    return None
                if ex.errno == errno.ECONNRESET:
                    raise EOF()
                raise
            finally:
                self.timeout = otimeout
        else:
            try:
                return self.sock.recv(readby)
            except socket.error as ex:
                if ex.errno == errno.ECONNRESET:
                    raise EOF()
                raise

    def send(self, data):
        return self.sock.send(data)

    def read(self, count, readby=8192, timeout=None, raiseEof=True):
        buff = []
        received = 0
        left = count

        if timeout:
            deadline = time.time() + timeout

        while left:
            try:
                data = self.recv(min(readby, left), timeout=deadline - time.time() if timeout else None)
            except socket.error as ex:
                if ex.errno == errno.ECONNRESET:
                    raise EOF()
                raise

            if data is None:
                # Timed out
                return None

            if data == '':
                if raiseEof:
                    raise EOF()
                else:
                    break

            left -= len(data)
            buff.append(data)

        return ''.join(buff)

    def write(self, data, timeout=None):
        length = len(data)
        totalSent = 0

        if timeout:
            deadline = time.time() + timeout
            otimeout = self.timeout

        try:
            while totalSent < length:
                try:
                    if timeout:
                        currentTimeout = deadline - time.time()
                        if currentTimeout < 0:
                            return None
                        self.timeout = currentTimeout

                    sent = self.send(data[totalSent:])
                except socket.error as ex:
                    if ex.errno == errno.EPIPE:
                        raise EOF()
                    if str(ex) == 'timed out':
                        return None
                    raise

                totalSent += sent
        finally:
            if timeout:
                self.timeout = otimeout

        return totalSent

    def readMsgpack(self, buff):
        unpacker = msgpack.Unpacker()

        while True:
            try:
                buf = self.recv(buff)
            except socket.error as ex:
                if ex.errno == errno.ECONNRESET:
                    raise EOF()
                if ex.errno == errno.EBADF:
                    return
                raise

            if buf == '':
                # No more data to process here, connection closed
                return

            unpacker.feed(buf)
            for data in unpacker:
                stop = yield data
                if stop:
                    yield 0
                    return

    def bind(self, host, port):
        assert not self.__binded
        self.sock.bind((host, port))
        self.__binded = True

    def connect(self, host, port, timeout=None):
        assert not self.__connected

        if timeout is not None:
            oldTimeout = self.timeout
            self.timeout = timeout

        try:
            if port is None:
                self.sock.connect(host)
            else:
                self.sock.connect((host, port))
            self.__connected = True
        finally:
            if timeout is not None:
                self.timeout = oldTimeout
