from __future__ import division


from collections import deque
import bisect


class TimeCounter(object):
    __slots__ = '_size', '_resolution', '_history', '_counters', '_countersList'

    class Bucket(object):
        __slots__ = 'num', 'bs', 'value'

        def __init__(self, num, bs, value=None):
            self.num = num        # block number
            self.bs = bs          # block start timestamp
            self.value = value    # total block length in seconds

        def __repr__(self):
            return '<Block %s>' % (
                ' '.join(
                    '%s=%s' % (k, getattr(self, k))
                    for k in self.__slots__
                )
            )

    class Counter(object):
        __slots__ = 'size', 'buckets', 'bucketsCount', 'value', 'last'

        def __init__(self, size, history=0):
            self.size = size
            self.buckets = deque(maxlen=size)
            self.bucketsCount = 0
            self.last = None
            self.value = 0

        def add(self, bucket):
            assert self.last != bucket
            if self.bucketsCount == self.size:
                self.update(-self.buckets[0].value)
            else:
                self.bucketsCount += 1

            self.buckets.append(bucket)
            self.last = bucket

        def update(self, difference):
            # Apply difference and ensure self.value will be >= 0
            self.value += difference
            if self.value < 0:
                self.value = 0

        def __repr__(self):
            return '<Counter size=%d value=%.4f>' % (self.size, self.value)

    def __init__(self, size=30, resolution=1.0):
        """
        Counter allows to count some float/int value with specific
        resolution in seconds.:

            1. Whole time (same as timestamp :))
            2. Percent of time spent between start() and stop()
            3. Percent of time spent between stop() and start()

        Args:
            size:
                max number of stat buckets to hold. For example, if you
                set size=30 and resolution=1, that means we will hold
                30 1-second buckets of stats and counter will tell you how
                much percent of time we spent in any period in last 30
                seconds

            resolution:
                length in seconds of each bucket
        """

        assert isinstance(size, int), 'Size argument must be an integer'
        assert isinstance(resolution, float), 'Resolution must be a float number'
        assert resolution > 0, 'Resolution must be a positive float number'

        self._size = size
        self._resolution = resolution

        self._history = []
        self._counters = {}
        self._countersList = []  # sorted list of counters

        self.addCounter(size)

    def _makeZeroBucket(self, num):
        return self.Bucket(
            num=num,
            bs=num * self._resolution,
            value=0.0
        )

    def _insertZeroBuckets(self, bnumFrom, bnumTo):
        for i in range(bnumFrom, bnumTo)[-self._size:]:
            bucket = self._makeZeroBucket(i)

            for counter in self._countersList:
                counter.add(bucket)
                counter.update(bucket.value)

    def addCounter(self, num):
        """
        Add fast counter of N last buckets average value

        Args:
            num:
                number of last buckets which average should be calculated
                right after new data available. This is quite fast, since we
                dont need to "sum()/len()" every time we need "last N seconds"
                stat
        """

        if num in self._counters:
            return False

        position = bisect.bisect(
            [c.size for c in self._countersList],
            num
        )
        counter = self.Counter(num)
        self._counters[num] = counter
        self._countersList.insert(position, counter)
        return True

    def _addBucket(self, bucket):
        for counter in self._countersList:
            counter.add(bucket)

    def _update(self, difference):
        for counter in self._countersList:
            counter.update(difference)

    def push(self, ts, value):
        raise NotImplementedError


class TimeSumCounter(TimeCounter):
    def push(self, ts, value):
        bnum = int(ts // self._resolution)
        bucket = self._counters[self._size].last

        if bucket is not None and bucket.num == bnum:
            bucket.value += value
        else:
            if bucket is not None:
                if bnum - bucket.num > 1:
                    self._insertZeroBuckets(bucket.num + 1, bnum)
                bs = bnum * self._resolution
            else:
                bs = ts

            bucket = self.Bucket(
                num=bnum,
                bs=bs,
                value=value
            )
            self._addBucket(bucket)

        self._update(value)
        return self.stat()

    def stat(self, size=None):
        if size is None:
            size = self._size

        try:
            counter = self._counters[size]
        except KeyError:
            counter = self.Counter(size=size)
            counter.buckets.extend(self._counters[self._size].buckets)
            counter.value = sum(b.value for b in counter.buckets)
            counter.last = counter.buckets[-1] if counter.buckets else None

        if counter.last is None:
            return 0

        return counter.value


class TimeIncrementCounter(TimeSumCounter):
    def push(self, ts):
        super(TimeIncrementCounter, self).push(ts, 1)


class TimeFrameCounter(TimeCounter):
    __slots__ = TimeCounter.__slots__ + ('_startTimestamp', )

    class Bucket(TimeCounter.Bucket):
        __slots__ = TimeCounter.Bucket.__slots__ + ('ts', 'te')

        def __init__(self, num, bs, ts, te=None, value=None):
            super(TimeFrameCounter.Bucket, self).__init__(num, bs, value)
            self.ts = ts          # last period start timestamp
            self.te = te          # last period end timestamp

    def __init__(self, size=30, resolution=1.0):
        super(TimeFrameCounter, self).__init__(size, resolution)
        self._startTimestamp = None

    def _makeZeroBucket(self, num):
        return self.Bucket(
            num=num,
            bs=num * self._resolution,
            ts=num * self._resolution,
            te=(num + 1) * self._resolution,
            value=0.0
        )

    def push(self, ts1, ts2):
        bnum = int(ts1 // self._resolution)
        bucket = self._counters[self._size].last  # grab last bucket we have

        difference = ts2 - ts1
        if bucket is not None and bucket.num == bnum:
            bucket.ts = ts1
            bucket.te = ts2
            bucket.value += difference
        else:
            if bucket is not None:
                if bnum - bucket.num > 1:
                    self._insertZeroBuckets(bucket.num + 1, bnum)
                bs = bnum * self._resolution
            else:
                bs = ts1

            bucket = self.Bucket(
                num=bnum,
                bs=bs,
                ts=ts1,
                te=ts2,
                value=difference
            )
            self._addBucket(bucket)

        self._update(difference)

        while bucket.value > self._resolution:
            bnum = bucket.num + 1
            bucket.te = bnum * self._resolution
            remainder = bucket.value - self._resolution
            bucket.value = self._resolution

            bucket = self.Bucket(
                num=bnum,
                bs=bnum * self._resolution,
                ts=bnum * self._resolution,
                te=ts2,
                value=remainder
            )

            self._addBucket(bucket)

        return self.stat()

    def start(self, ts):
        assert self._startTimestamp is None
        self._startTimestamp = ts

    def stop(self, ts):
        assert self._startTimestamp is not None
        try:
            return self.push(self._startTimestamp, ts)
        finally:
            self._startTimestamp = None

    def stat(self, size=None):
        # By default return whole history average
        if size is None:
            size = self._size

        try:
            counter = self._counters[size]
        except KeyError:
            counter = self.Counter(size=size)
            counter.buckets.extend(self._counters[self._size].buckets)
            counter.value = sum(b.value for b in counter.buckets)
            counter.last = counter.buckets[-1] if counter.buckets else None

        if counter.last is None:
            return 0

        worktime = counter.value
        livetime = counter.buckets[-1].te - counter.buckets[0].bs

        if livetime <= 0:
            return 0

        return worktime / livetime
