class Trigger(object):
    VALID_NAMES = ['warn', 'error']

    __slots__ = ['name', 'min', 'max']

    def __init__(self):
        for n in self.__slots__:
            setattr(self, n, None)

    def __repr__(self, x=None):
        sfmt = lambda x: '%.4g' % x if x is not None else 'None'
        fmt = '%s(%s <' + ('' if x is None else ' %.4g ' % x) + '> %s)'
        return 'EmptyTrigger' if self.empty() else fmt % (self.name, sfmt(self.min), sfmt(self.max))

    def validate(self, s):
        if not self.name:
            if not self.empty(['min', 'max']):
                raise Exception('Unnamed trigger in %r' % s)
        else:
            if self.name not in self.VALID_NAMES:
                raise Exception('Invalid trigger name %r in %r' % (self.name, s))
            if self.empty(['min', 'max']):
                raise Exception('Named trigger without any limits in %r.' % s)
            if self.min is not None and self.max is not None and self.min >= self.max:
                raise Exception('Lower limit is greater or equal to upper limit in %r.' % s)

    def empty(self, attrs=None):
        for a in attrs if attrs else self.__slots__:
            if getattr(self, a) is not None:
                return False
        return True

    def check(self, x):
        return self.min and x < self.min or self.max and x > self.max

    @staticmethod
    def parse(s, start=0):
        self = Trigger()
        op = None
        word = ''

        s = s[start:]
        l = len(s)
        at = 0

        def onOp(op, word, inv=False):
            if not op:
                return
            try:
                v = float(word)
            except ValueError as ex:
                raise Exception(
                    'Unable to parse symbol %r at position %d of %r as numeric: %s' %
                    (word, at, s[start:], str(ex))
                )

            chk = '<' if not inv else '>'
            attr = 'min' if op == chk else 'max'
            if getattr(self, attr):
                    raise Exception('%r limit specified twice at position %d of %r' % (attr, at, s))
            setattr(self, attr, v)

        while at < l:
            c = s[at]
            at += 1
            if c in ', x':   # Ignore characters
                continue
            elif c == '(':
                if not self.empty() or not word:
                    raise Exception('Invalid character %r at position %d of %r' % (c, at, s))
                self.name = word
            elif c == ')':
                if word:
                    onOp(op, word)
                elif (self.max and self.min) is None:
                    self.max, self.min = self.min, self.max
                self.validate(s)
                return self, at + start
            elif c in '<>':
                if word:
                    onOp(c, word)
                op = c
            else:
                word += c
                continue
            word = ''

        if not self.empty():
            raise Exception('Unterminated trigger %r declaration: %r' % (self.name, s))
        return None, at

    @classmethod
    def fromString(cls, s):
        res = []
        l = len(s)
        at = 0
        while at < l:
            t, at = cls.parse(s, at)
            if not t:
                break
            res.append(t)

        if len(res) > 2:
            raise Exception('More than two triggers declaration: %r' % s)
        if len(res) > 1 and res[0].name == res[1].name:
            raise Exception('Declaration of two triggers of the same name: %r' % s)
        return res
