// Global variable, which handles last server's response.
var server_data = {};
// Globally stored chart and bar objects.
var charts = [], bars = [], empty = [];
// Charts meta info for data
const chartMeta = {
    dead:   {label: 'Dead',     color: $.color.parse('#377EB8').scale('a', 0.6).toString()},
    running:{label: 'Running',  color: $.color.parse('green').scale('a', 0.8).toString()},
    ok:     {label: 'OK',       color: $.color.parse('#4DAF4A').scale('a', 0.75).toString()},
    wrong:  {label: 'Wrong',    color: $.color.parse('#E41A1C').scale('a', 0.8).toString()},
    error:  {label: 'Error',    color: $.color.parse('magenta').scale('a', 0.75).toString()},
    warn:   {label: 'Warning',  color: $.color.parse('blue').scale('a', 0.75).toString()},
    missing:{label: 'Missing',  color: $.color.parse('yellow').scale('a', 0.5).toString()}
};
var data_source = '';
// Update color codes for bars.
for (var key in chartMeta) {
    if (chartMeta.hasOwnProperty(key))
        chartMeta[key].colorLight = $.color.parse(chartMeta[key].color).scale('a', 1.75);
}

// The page is ready for data updates flag.
var ready = null;
// Base URL
var baseurl = location.protocol + '//' + location.hostname + (location.port ? ':' + location.port : '')
// Data provider URL
var ajaxurl = baseurl + location.pathname + '?json';
// Detailed hosts list base URL.
var versionsurl = baseurl + '/versions';

// Attach the `equals()` method to Array's prototype to call it on any array.
// Implements shallow/deep check for arrays equalancy.
Array.prototype.equals = function (array, deep) {
    // if the other array is a falsy value, return
    if (!array)
        return false;

    // compare lengths - can save a lot of time
    if (this.length != array.length)
        return false;

    for (var i = 0; i < this.length; i++) {
        // Check if we have nested arrays
        if (deep && this[i] instanceof Array && array[i] instanceof Array) {
            // recurse into the nested arrays
            if (!this[i].equals(array[i]))
                return false;
        } else if (this[i] != array[i]) {
            // Warning - two different object instances will never be equal: {x:20} != {x:20}
            return false;
        }
    }
    return true;
};

Highcharts.setOptions({
    global: {
        useUTC: false
    }
});

function placeChart(place) {
    var place = $('#' + place);
    var chart = $.plot(place, [], {
        series: {
            lines: {show: true, hoverable: false, clickable: false, fill: true, lineWidth: 1, zero: true},
            points: {show: false},
            shadowSize: 0,
            stack: true,
        },
        xaxis: {
            mode: 'time',
            timezone: 'browser'
        },
        yaxis: {
            autoscaleMargin: null
        },
        grid: {
            hoverable: true
        },
        legend: {
            position: 'nw',
            noColumns: 1,
            sorted: 'reverse'
        },
        tooltip: true,
        crosshair: {mode: 'x'}
    });

    var updateLegendTimeout = null;
    var latestPosition = null;

    function updateLegend() {
        var legends = place.find('.legendLabel');
        legends.each(function () {
            // fix the widths so they don't jump around
            var rwidth = $(this).width();
            $(this).parents('.legend').children('div').css('width', rwidth + 20);
        });

        updateLegendTimeout = null;

        var pos = latestPosition;

        var axes = chart.getAxes();
        if (pos.x < axes.xaxis.min || pos.x > axes.xaxis.max ||
            pos.y < axes.yaxis.min || pos.y > axes.yaxis.max) {
            return;
        }

        var i, j, dataset = chart.getData();
        for (i = 0; i < dataset.length; ++i) {

            var series = dataset[i];

            // Find the nearest points, x-wise

            for (j = 0; j < series.data.length; ++j) {
                if (series.data[j][0] > pos.x) {
                    break;
                }
            }

            // Now Interpolate

            var y,
                p1 = series.data[j - 1],
                p2 = series.data[j];

            if (p1 == null) {
                y = p2[1];
            } else if (p2 == null) {
                y = p1[1];
            } else {
                y = p1[1] + (p2[1] - p1[1]) * (pos.x - p1[0]) / (p2[0] - p1[0]);
            }

            var newlabel = series.label.replace(/\(\d+\)/, '(' + y.toFixed(0) + ')');
            legends.eq(dataset.length - i - 1).text(newlabel);
        }
    }

    place.bind("plothover",  function (event, pos, item) {
        latestPosition = pos;
        if (!updateLegendTimeout) {
            updateLegendTimeout = setTimeout(updateLegend, 50);
        }
    });

    return chart;
}

function placeBar(index, place) {
    var series = [];
    var keys = Object.keys(chartMeta);
    for (var i = keys.length; i > 0; --i)
        series.push({ name: '', data: [], color: chartMeta[keys[i - 1]].colorLight });
    var chart = new Highcharts.Chart({
        chart: {
            type: 'bar',
            renderTo: place,
            spacingTop: 5,
            spacingBottom: 0,
            spacingLeft: 5,
            spacingRight: 5,
            animation: false
        },
        credits: { enabled: false },
        tooltip: {
            //shared: true,
            useHTML: true,
            headerFormat: '',
            followPointer: true,
            footerFormat: '<i class="footnote">' +
                'Click the bar to toggle zoomable details chart.<br/>' +
                'Click on the label to view detailed hosts list page.</i>'
        },
        title: {
            text: 'Stacked bar chart',
            align: 'left'
        },
        xAxis: {
            categories: [''],
            labels: { enabled: false }
        },
        yAxis: {
            min: 0,
            title: { text: '' },
            labels: { align: 'left', y: 2, x: 2 }
        },
        legend: {
            floating: true,
            align: 'right',
            verticalAlign: 'top', y: 15,
            reversed: true
        },
        plotOptions: {
            series: {
                stacking: 'percent'
            },
            bar: {
                events: {
                    click: function(e) { toggleChart(index); },
                    legendItemClick: function(event) {
                        console.log(new Date(), 'this:'+this.index+'   '+this.data[0].y+'   ');
                        // open new window if we have something to show
                        if (this.data[0].y){
                            window.open(versionsurl +
                                '?rule=' + this.chart.name + '&source=' + data_source + '&type=' +
                                chartMeta[Object.keys(chartMeta).slice(-this.index - 1)[0]].label);
                        }
                        return false;
                    }
                },
                cursor: 'pointer'
            },
        },
        series: series
    });
    chart.showLoading();
    return chart;
}

function addChartPoint(chart, ts, points, redraw, append) {
    for (var i = 0; i < points.length; ++i)
        chart.series[i].addPoint(ts ? [ts, points[i]] : points[i], i != points.length - 1 ? false : redraw, append);
}

function updateChartMeta(chart, serie, points, initial) {
    chart.setTitle({text: '<b>' + serie.name + '</b> [' + serie.task + ']'});
    for (var i = 0; i < points.length; ++i)
        chart.series[i].update({
            name: chartMeta[Object.keys(chartMeta).slice(points.length - i - 1)[0]].label + ' (' + points[i] + ')'
        });
    if (initial) {
        chart.name = serie.name;
        chart.hideLoading();
    }
}

function updateChart(chart, ttable, serie, initial) {
    var i, pt = null;
    var data = chart.getData();
    if (data.length == 0) {
        for (var key in chartMeta)
            data.push({
                data:  [],
                color: chartMeta[key].color,
                label: chartMeta[key].label + ' (0)',
                lines: {fillColor: chartMeta[key].color}
            });
    }

    var keys = Object.keys(chartMeta);
    for (i = 0; i < serie.data.length; i++) {
        pt = serie.data[i];
        var dt = new Date(ttable[i] * 1000);
        for (var k = 0; k < keys.length; ++k)
            data[k].data.push([dt, pt[keys[k]]]);
    }

    for (i = 0; i < data.length; i++) {
        var lastval = data[i].data[data[i].data.length - 1][1];
        data[i].label = data[i].label.replace(/\(.*\)/, '(' + lastval + ')')
    }

    for (i = 0; !initial && i < data.length; i++)
        data[i].data.splice(0, serie.data.length);

    chart.setData(data);
    chart.setupGrid();
    chart.draw();
}

function updateBar(chart, serie, initial) {
    var keys = Object.keys(chartMeta);
    var points = [], pts = serie.data.slice(-2);
    for (var i = 0; i < pts.length; ++i) {
        var pt = points[i] = [];
        for (var k = keys.length; k > 0; --k)
            pt.push(pts[i][keys[k - 1]]);
    }
    if (!initial && points[0].equals(points[1])) return; // No need to update anything
    addChartPoint(chart, null, points[1], !initial, !initial);
    updateChartMeta(chart, serie, points[1], initial);
}

function drawBars() {
    var i;
    // Ok, now we can enable charts there.
    for (i = 0; i < bars.length; ++i)
        bars[i] = placeBar(i, 'skyprogress' + i);
    // And finally update them all with initial value.
    for (i = 0; i < bars.length; ++i)
        updateBar(bars[i], server_data.series[i], true);
    return false;
}

function toggleChart(index) {
    var id = 'skyversions' + index;
    var ph = $('#' + id);
    if (charts[index]) {
        ph.hide("blind", {}, 500, function() {
            charts[index].shutdown();
            ph.find().remove()
            charts[index] = null;
            var name = server_data.series[index].name;
            if (window.location.hash.substring(1) == name) {
                var anyopened = false
                for (var i = 0; i < charts.length; ++i) {
                    if (charts[i]) {
                        window.location.hash = server_data.series[i].name;
                        anyopened = true
                        break;
                    }
                }
                if (!anyopened) {
                    window.location.hash = '';
                }
            }
        });
    } else {
        var chart = charts[index] = placeChart(id);
        ph.show("blind", {}, 500, function() {
            var serie = server_data.series[index];
            window.location.hash = serie.name;
            updateChart(chart, server_data.ttable, serie, true);
        });
    }
}

function onDataUpdate(update) {
    var i, profile = '';
    if (server_data.series.length != update.series.length) {
        console.log(
            'Amount of rules changed from ' + server_data.series.length +
            ' to ' + update.series.length + '. Reloading the page.'
        );
        window.location.reload();
    } else if (server_data.confid != update.confid) {
        console.log('Configuration ID changed to ' + update.confid + '. Updating the page. Source:' + data_source);
        $("#loading_dialog").loading("loadStart");
        ready = false;
        for (i = 0; i < bars.length; ++i) {
            if (bars[i]) bars[i].destroy();
            if (charts[i]) charts[i].shutdown();
        }
        init(data_source);
        return;
    }
    var amount = update.ttable.length;
    if (!amount || !ready) return;

    console.log(new Date(), 'Partial data update. ' + amount + ' entr(y,ies) given.');

    var now = new Date().getTime();
    // Drop "outdated" time table entries and add fresh ones
    server_data.ttable.splice(0, amount);
    server_data.ttable = server_data.ttable.concat(update.ttable);
    // Do the same for all series.
    var bar_points = [];
    for (i = 0; i < server_data.series.length; ++i) {
        var upd = update.series[i];
        var serie = server_data.series[i];
        if (amount > 1) {
            bar_points[i] = jQuery.extend({}, upd);
            bar_points[i].data = [serie.data.slice(-1)[0], upd.data.slice(-1)[0]];
        }

        serie.data.splice(0, amount);
        var data = serie.data.concat(upd.data);
        serie = jQuery.extend({}, upd);
        serie.data = data;
        server_data.series[i] = serie;
        if (amount == 1) bar_points[i] = serie;
    }
    var prev = [now, now = new Date().getTime()][0];
    profile += now - prev;

    for (i = 0; i < charts.length; ++i) {
        if (charts[i]) updateChart(charts[i], update.ttable, update.series[i], false);
    }
    prev = [now, now = new Date().getTime()][0];
    profile += '/' + (now - prev);
    for (i = 0; i < bars.length; ++i) {
        if (bars[i]) updateBar(bars[i], bar_points[i], false);
    }
    prev = [now, now = new Date().getTime()][0];
    profile += '/' + (now - prev);
    $('#profile').html(profile);
}

function autoUpdate() {
    $.ajax({
        url: ajaxurl + "&ts=" + server_data.ttable.slice(-1)[0],
        headers: {Accept: "application/json; charset=utf-8"}
    }).success(onDataUpdate);
}

function init(source) {
    console.log('AJAX base URL: ', ajaxurl);
    var profile = '', now = new Date().getTime();
    data_source = source;
    $.ajax({
        async: false,
        url: ajaxurl,
        headers: {Accept: "application/json; charset=utf-8"},
    }).success(function(data) {
        console.log(new Date(), 'Initial chart update. ' + data.ttable.length + ' entries given.');
        console.log(new Date(), 'Binding to ' + data.confid + ' configuration identifier.');
        server_data = data;
    });

    // Determine name of detailed chart to open
    var ancor = window.location.hash.length ? window.location.hash.substring(1) : null;

    var prev = [now, now = new Date().getTime()][0];
    profile += now - prev;
    // Create empty array of lengths equal to amount of rules.
    empty[server_data.series.length - 1] = null;
    bars = jQuery.extend([], empty);
    // Prepare the correct layout
    var i, ph = $('#charts');
    for (i = 0; ready == null && i < server_data.series.length; ++i) {
        var name = server_data.series[i].name;
        if (name == ancor) charts[i] = true;
        ph.html(
            ph.html() +
            '<div id="skyprogress' + i +
                    '" style="height: 110px; width: 940px; margin-bottom: 5px"></div>' +
            '<div id="skyversions' + i +
                    '" style="height: 165px; width: 940px; margin-bottom: 5px; display: none"><svg /></div>\n'
        );
    }

    // Show requested detailed charts.
    for (i = 0; i < charts.length; ++i) {
        if (charts[i]) {
            charts[i] = null;
            toggleChart(i);
        }
    }

    prev = [now, now = new Date().getTime()][0];
    profile += '/' + (now - prev);

    // Draw all required bars.
    drawBars();
    prev = [now, now = new Date().getTime()][0];
    profile += '/' + (now - prev);
    $('#profile').html(profile);

    if (ancor) {
        for (i = 0; i < charts.length; ++i)
            if (charts[i]) break;
        $('html, body').animate({scrollTop: $('#skyprogress' + i).offset().top}, 1000);
    }

    // That's all actually.
    if (ready == null) setInterval(autoUpdate, 2500);
    $("#loading_dialog").loading("loadStop");
    ready = true;
}
