import copy
import time
import gzip
import traceback
import itertools
import collections
import math
import datetime as dt
import json

import py
import msgpack
import socket
import gevent
import gevent.queue
import gevent.event
try:
    import gevent.coros as coros
except ImportError:
    import gevent.lock as coros
import six
from collections import defaultdict, OrderedDict

from kernel import util
from library.sky.hostresolver.resolver import Resolver


class Entity(object):
    """
    The class is designed as simple slotted data entry storage with an ability to fill in
    slots available via keyword arguments in its constructor. In case of no value provided
    to the constructor, it will be fetched out from `__defs__` array (which should be of
    the same length as `__slots__`).
    """
    __slots__ = []
    __defs__ = []

    def __init__(self, *args, **kwargs):
        for i, attr in enumerate(self.__slots__):
            _def = self.__defs__[i]
            if i < len(args):
                val = args[i]
            elif attr in kwargs:
                val = kwargs[attr]
            else:
                val = copy.copy(_def)
                _def = None  # Avoid double checking
            setattr(self, attr, val if _def is None or isinstance(val, _def.__class__) else _def.__class__(val))

    def __repr__(self):
        return self.__class__.__name__ + repr(dict(self))

    def __iter__(self):
        for attr in self.__slots__:
            yield (attr, getattr(self, attr))

    def itervalues(self):
        for attr in self.__slots__:
            yield getattr(self, attr)

    def copy(self):
        return self.__class__([v for _, v in self])


class LinkedExited(Exception):
    pass


class LinkedCompleted(LinkedExited):
    """Raised when a linked greenlet finishes the execution cleanly"""

    msg = "%r completed successfully"

    def __init__(self, source):
        assert source.ready(), source
        assert source.successful(), source
        LinkedExited.__init__(self, self.msg % source)


class LinkedKilled(LinkedCompleted):
    """Raised when a linked greenlet returns GreenletExit instance"""

    msg = "%r returned %s"

    def __init__(self, source):
        try:
            result = source.value.__class__.__name__
        except:
            result = str(source) or repr(source)
        LinkedExited.__init__(self, self.msg % (source, result))


class LinkedFailed(LinkedExited):
    """Raised when a linked greenlet dies because of unhandled exception"""

    msg = "%r failed with %s"

    def __init__(self, source):
        exception = source.exception
        try:
            excname = exception.__class__.__name__
        except:
            excname = str(exception) or repr(exception)
        LinkedExited.__init__(self, self.msg % (source, excname))


class GreenletLink(object):
    __slots__ = ['greenlet']

    def __init__(self, greenlet=None):
        self.greenlet = greenlet if greenlet is not None else gevent.getcurrent()

    def __call__(self, source):
        if source.successful():
            if isinstance(source.value, gevent.GreenletExit):
                e = LinkedKilled(source)
            else:
                e = LinkedCompleted(source)

        else:
            e = LinkedFailed(source)

        self.greenlet.throw(e)


class Pool(object):
    def __init__(self, log, appStopper, **kwargs):
        for k, v in six.iteritems(kwargs):
            setattr(self, k, v)
        self._app_stopper = appStopper
        self._log = log.getChild(self.__module__.rsplit('.', 1)[-1]).getChild(self.__class__.__name__)

    def __getitem__(self, item):
        return getattr(self, item)

    def _watchdogLoop(self, log, stopper):
        log.info("Background tasks pool watchdog started.")
        greenlets = [v._worker for k, v in six.iteritems(self.__dict__) if isinstance(v, BgTask) and v._worker is not None]
        failed_to_start = [k for k, v in six.iteritems(self.__dict__) if isinstance(v, BgTask) and v._worker is None]
        try:
            if failed_to_start:
                raise Exception("Greenlets failed to start: %s" % (failed_to_start,))
            for gt in greenlets:
                gt.link(GreenletLink())
            gevent.getcurrent().join()  # Sleep forever!
        except gevent.GreenletExit:
            log.info("Watchdog stopped.")
        except Exception as ex:
            log.error("Some of background tasks stopped unexpectedly (%r). Stopping the application.", ex)

        for gt in greenlets:
            gt.unlink(None)

        self._do('stop')
        stopper()

    def ping(self):
        return not self._watchdog.ready()

    def _do(self, meth):
        return dict((k, getattr(v, meth)()) for k, v in six.iteritems(self.__dict__) if isinstance(v, BgTask))

    def start(self):
        self._do('start')
        self._watchdog = gevent.spawn(
            self._watchdogLoop,
            self._log,
            self._app_stopper
        )

    def stop(self):
        self._watchdog.kill()

    def join(self):
        return self._do('join')

    def running(self):
        return self._do('running')


class BgTaskStoppingException(Exception):
    def _init__(self):
        super(BgTaskStoppingException, self).__init__('Forcedly stopped task')


class BgTask(object):
    """
    Background task abstract base.
    """
    def __init__(self, log, period, name=None):
        self.log = log.getChild(self.__module__.rsplit('.', 1)[-1]).getChild(
            name if name else self.__class__.__name__
        )
        self._period = period

        self._worker = None
        self._queue = gevent.queue.Queue()
        self._stopping = False
        self.name = name

    def start(self):
        self._worker = gevent.spawn(self._workerLoop)
        return self

    def stop(self):
        self._stopping = True
        self._queue.put(None)
        return self

    def join(self):
        self._worker.join()
        return self

    def running(self):
        return not self._worker.ready()

    def _do(self, first=False, forced=False):
        raise NotImplementedError

    def _workerLoop(self):
        self.log.info('Background thread started.')
        ev = None
        try:
            self._do(True, bool(ev))
            while True:
                try:
                    ev = self._queue.get(timeout=self._period)
                    if ev is None:
                        break
                except gevent.queue.Empty:
                    ev = False
                self._do(False, bool(ev))
                if ev:
                    ev.set()
        except BgTaskStoppingException:
            self.log.info('Task forcedly stopped')
        except Exception:
            self.log.error('Unhandled exception: %s', traceback.format_exc())

        # Release any pending waiters
        try:
            while True:
                if ev:
                    ev.set()
                ev = self._queue.get_nowait()
        except gevent.queue.Empty:
            pass

        self.log.info('Background thread stopped.')

    def force(self):
        if self._worker.ready():
            return
        ev = gevent.event.Event()
        self._queue.put(ev)
        ev.wait()


class WatchdogTask(BgTask):
    def __init__(self, log):
        super(WatchdogTask, self).__init__(log, 1, 'Watchdog')
        self._last = time.time()

    def _do(self, first=False, forced=False):
        diff = time.time()-self._last
        self._last = time.time()
        if diff > 3:
            self.log.error('TICK: {}'.format(diff))


class RulezFetcher(BgTask):
    REQUEST_TIMEOUT = 30

    class Entry(Entity):
        __slots__ = ['name', 'desc', 'task', 'version', 'hosts']
        __defs__ = (None, None, None, None, set())

    def __init__(self, log, url, service_name):
        self._data = []
        self.mtime = None
        self.conf_id = None
        self.url = url
        super(RulezFetcher, self).__init__(log, 30, service_name + '_fetcher')

    def _do(self, first=False, forced=False):
        if forced:
            self.log.info('Fetching rules data forcedly.')
        try:
            opener = six.moves.urllib.request.build_opener()
            opener.addheaders = [('Accept-Encoding', 'gzip'),
                                 ('User-Agent', 'SkynetHeartbeat/1.0')]
            if self.mtime:
                opener.addheaders.append(('If-Modified-Since', self.mtime))
            self.log.debug('Requesting cloud API via %r', self.url)
            req = opener.open(self.url, timeout=self.REQUEST_TIMEOUT)
        except Exception as ex:
            if isinstance(ex, six.moves.urllib_error.HTTPError) and ex.code == 304:  # "Not Modified"
                # self.log.debug('Configuration has not been modified since %r.', self.mtime)
                return

            self.log.warn('Unable to fetch rules data: %r', str(ex))
            return

        data = req.read()
        size = util.size2str(len(data))
        if req.info().get('Content-Encoding') == 'gzip':
            data = gzip.GzipFile(fileobj=six.BytesIO(data)).read()
            size += ' (%s)' % util.size2str(len(data))
        data = msgpack.loads(data)
        self.mtime = req.headers.get('Last-Modified')
        self.conf_id = req.headers.get('X-Ya-Cms-Conf-Id')

        self._data = [
            self.Entry(
                x['name'],
                x['desc'],
                x['sandbox_task_name'],
                x['resolved']['resource']['UNKNOWN']['attrs'].get(
                    'version',
                    x['resolved']['resource']['svn_url'].rsplit('/', 2)[-1] if x['resolved']['resource']['svn_url']
                    else x['resolved']['resource']['UNKNOWN']['attrs'].get('resource_version', 'MISSING')
                ),
                set(x['resolved']['hosts'])
            ) for x in data
        ]
        self.log.info(
            'Got %s response of %d rules of %d hosts total.',
            size, len(self._data), sum(len(x.hosts) for x in self._data)
        )

    @property
    def rules(self):
        return self._data


class RulezHistory(BgTask):
    DUMP_VERSION = 3
    DEFAULT_RULE_NAME = 'DEFAULT'

    class StatsEntry(Entity):
        # The name-to-attribute matches human-readable name of statistics entry to attribute of the data entry.
        NAME2ATTR = [
            ('OK', 'ok'),
            ('Warning', 'warn'),
            ('Error', 'error'),
            ('Wrong', 'wrong'),
            ('Dead', 'dead'),
            ('Missing', 'missing'),
            ('Running', 'running')
        ]
        __slots__ = ['ok', 'warn', 'error', 'wrong', 'dead', 'missing', 'running']
        __defs__ = (0, ) * 7

        @staticmethod
        def names():
            return [x[0] for x in RulezHistory.StatsEntry.NAME2ATTR]

    class DataEntry(Entity):
        __slots__ = ['rule', 'order', 'hosts', 'stats']
        __defs__ = (None, 0, {}, [])
        __name2attr = None
        __statattrs = None

        def counts(self):
            stattrs = self.__class__.__statattrs
            if not stattrs:
                stattrs = self.__class__.__statattrs = \
                    dict.fromkeys((x[1] for x in RulezHistory.StatsEntry.NAME2ATTR), 0)
            ret = stattrs.copy()
            for v in six.itervalues(self.hosts):
                ret[v if v else 'missing'] += 1
            return dict(ret)

        def getattrbyname(self, name):
            n2a = self.__class__.__name2attr
            if not n2a:
                n2a = self.__class__.__name2attr = dict(
                    (x[0].lower(), x[1]) for x in RulezHistory.StatsEntry.NAME2ATTR
                )
                n2a['missing'] = None
            _type = n2a[name.lower()]
            if _type:
                _type = _type.lower()
            return set(h for h, t in six.iteritems(self.hosts) if t == _type)

        def clear(self):
            self.hosts = dict.fromkeys(self.rule.hosts)

        def getOrder(self):
            return self.order

    def __init__(self, cfg, log, wdir, fetcher, ver_path, new_ver_path, service_name, service_full_name, event_callback):
        super(RulezHistory, self).__init__(log, cfg.lacmus.history.period, service_name + '_history')

        self._lacmus_cfg = cfg.lacmus.history
        self._juggler_cfg = cfg.golem.juggler
        self._dumped = 0
        self._conf_id = None
        self._data = {}
        self._ttable = collections.deque(maxlen=self._lacmus_cfg.max_stack_length)
        self._dataLock = coros.RLock()
        self._fetcher = fetcher
        self._ver_path = ver_path
        self._new_ver_path = new_ver_path
        self._dump = py.path.local(wdir).join(self._lacmus_cfg.dump_file + '.' + '.'.join(ver_path))
        self._event_name = service_name + '_service_state'
        self._event_callback = event_callback
        self._service_full_name = service_full_name

        self._update_time = 0
        self._update_count = 0.0
        self._update_deadline = 0

        self.log.info('{}: {} {}'.format(service_name, new_ver_path, ver_path))
        try:
            now = time.time()
            dump = msgpack.load(self._dump.open('rb'))
            if dump['version'] != self.DUMP_VERSION:
                raise Exception('Dump file version: %r found, %r expected.' % (dump['version'], self.DUMP_VERSION))
            self._conf_id = dump['conf_id']
            self._ttable.extend(dump['ttable'])
            for k, v in six.iteritems(dump['data']):
                entry = self._data[k] = self.DataEntry(RulezFetcher.Entry(**v['rule']), **v)
                entry.stats = collections.deque(
                    (self.StatsEntry(**st) for st in v['stats']),
                    maxlen=self._lacmus_cfg.max_stack_length
                )
            self._dumped = now
            self.log.info(
                'Data dump of %s old has been successfully loaded from %r',
                util.td2str(now - self._dump.stat().mtime), self._dump.strpath
            )
        except Exception as ex:
            self.log.warn('Unable to load previously stored dump: %s. Filling stats with zeros.', str(ex))

        diff = self._lacmus_cfg.max_stack_length - len(self._ttable)
        if diff:
            self.log.info("Data dump missing %r entries. Filling them with dummies", diff)
            now = int(time.time())
            self._ttable.extend(ts for ts in range(now - self._lacmus_cfg.period * diff, now, self._lacmus_cfg.period))
            for entry in six.itervalues(self._data):
                entry.stats.extendleft(self.StatsEntry() for _ in range(diff))

    def get_version(self, info):
        tmp_info = ''
        for path in [self._new_ver_path, self._ver_path]:
            if not path:
                continue

            tmp_info = info
            for p in path:
                tmp_info = tmp_info.get(p, '')
                if not tmp_info:
                    break

            if tmp_info:
                # we found version
                break

        return tmp_info

    def _do_dump(self):
        stts = time.time()
        self._dump.ensure()
        entry2dict = lambda x, setconv=list: dict((k, v if not isinstance(v, set) else setconv(v)) for k, v in x)
        dump = {'version': self.DUMP_VERSION, 'conf_id': self._conf_id, 'ttable': list(self._ttable)}
        data = dump['data'] = {}
        for k, v in six.iteritems(self._data):
            entry = data[k] = entry2dict(v, setconv=lambda _: [])
            entry['rule'] = entry2dict(v.rule)
            entry['stats'] = [dict(st) for st in v.stats]
        try:
            msgpack.dump(dump, self._dump.open('wb'))
            self.log.debug('Full state dumped in %.3gs.', time.time() - stts)
        except BaseException as ex:
            self.log.error('Failed to dump current state: {}'.format(ex))

    def _do(self, first=False, forced=False):
        # do nothing here
        pass

    def start_update_data(self, first, rebuild):
        with self._dataLock:
            if first:
                self._fetcher.force()

            no_change = True

            # Update our own copy of rules
            if self._conf_id != self._fetcher.conf_id:
                self.log.info('Configuration has been switched from %r to %r', self._conf_id, self._fetcher.conf_id)
                actual_rules = set()
                for i, r in enumerate(self._fetcher.rules):
                    actual_rules.add(r.name)
                    entry = self._data.get(r.name)
                    if r.name not in self._data:
                        # New rule observed, fill it with empties.
                        entry = self._data[r.name] = self.DataEntry()
                        entry.stats = collections.deque(
                            (self.StatsEntry() for _ in range(self._lacmus_cfg.max_stack_length + 1)),
                            maxlen=self._lacmus_cfg.max_stack_length
                        )
                    entry.rule, entry.order = r, i

                # Exclude outdated rules
                for name in set(self._data.keys()) - actual_rules:
                    del self._data[name]
                # Update configuration ID, which this dataset bound to.
                self._conf_id = self._fetcher.conf_id
                if not first and not rebuild:
                    # And request full data reload if it's not yet
                    no_change = False
                    rebuild = True

            self._update_deadline = dt.datetime.now() - dt.timedelta(seconds=self._lacmus_cfg.alive_threshold)
            if rebuild:
                # Reset all hosts collections - they will be fetched out from the database.
                for entry in six.itervalues(self._data):
                    entry.clear()

            self._update_count = 0
            self._update_time = 0.0
            return no_change

    def update_data(self, doc):
        with self._dataLock:
            start_time = time.time()

            host = doc['host']
            ver = self.get_version(doc['skynet'])
            if not ver:
                return

            exitcode = doc.get('gosky', {}).get('exitcode', 0)
            services_ok, services_running = self._check_services(doc['skynet'])

            entry = next(
                (e for e in six.itervalues(self._data) if host in e.rule.hosts),
                self._data.get(self.DEFAULT_RULE_NAME, None)
            )

            if not entry:
                return

            if doc['last_update'] < self._update_deadline:
                entry.hosts[host] = 'dead'
            else:
                # compare current and expected versions
                # (in case of service we are searching for version in long service description)
                if ver != entry.rule.version and entry.rule.version not in ver:
                    # version is not correct
                    # if gosky exitcode says OK or 'some services are not started' -> show host's state as Wrong
                    # else show as Error, because we were not able to update skynet
                    if exitcode == 0 or exitcode == 12:
                        entry.hosts[host] = 'wrong'
                    else:
                        entry.hosts[host] = 'error'
                else:
                    # version is correct, so we have to decide whether host's state is Running, OK or Warning
                    if services_running:
                        entry.hosts[host] = 'running'
                    else:
                        entry.hosts[host] = 'ok' if services_ok else 'warn'

            self._update_count += 1
            self._update_time += time.time() - start_time

    def end_update_data(self):
        with self._dataLock:
            now = time.time()
            # Push new entry into each common time table.
            self._ttable.append(int(now))

            # Push new entry into each rule's statistics collection.
            for e in six.itervalues(self._data):
                e.stats.append(self.StatsEntry(**e.counts()))

            self.log.debug(
                'Historical data updated: %d records processed in %.3gs.',
                self._update_count,
                self._update_time)

            # Dump the data collected in case of full reload performed
            if self._dumped + self._lacmus_cfg.dump_every < now:
                self._dumped = now
                self._do_dump()

            failed_rules = []
            for e in six.itervalues(self._data):
                if not len(e.stats) or not len(e.rule.hosts):
                    continue
                stats = e.stats[-1]
                warn = float(stats.warn)/float(len(e.rule.hosts))
                if warn > self._juggler_cfg.warn_threshold and \
                        len(e.rule.hosts) >= self._juggler_cfg.min_hosts_count:
                    failed_rules.append('Failure in {} - {:.4}% hosts ({}/{})'.format(
                        e.rule.name,
                        warn * 100,
                        stats.warn,
                        len(e.rule.hosts)
                    ))

            if failed_rules:
                event_description = '\n'.join(failed_rules)
                state = 'critical'
            else:
                event_description = ''
                state = 'ok'

            # send event
            self._event_callback(self._event_name, event_description, state)

    def _check_services(self, data):
        # try new format
        try:
            if self._service_full_name != 'version':
                state = data['skycore']['services'][self._service_full_name]['state']
                return state in ['PRERUNNING', 'RUNNING', 'PREFAIL'], state == 'RUNNING'

            services = data['skycore']['services']
            return (
                all(service['state'] in ['PRERUNNING', 'RUNNING', 'PREFAIL']
                    for service in six.itervalues(services)),
                all(service['state'] == 'RUNNING' for service in six.itervalues(services))
            )

        except (KeyError, TypeError):
            # try old format
            try:
                if self._service_full_name != 'version':
                    return data['srvmngr']['services'][self._service_full_name], False

                services = data['srvmngr']['services']
                return all(six.itervalues(services)), False

            except (KeyError, TypeError):
                return False, False

    @property
    def storage(self):
        return self._data

    def data(self, ts=None):
        stts = time.time()
        with self._dataLock:
            delay = time.time() - stts
            if delay > 2:
                self.log.error('{} wait for {}'.format(self.name, delay))

            ts = int(ts) if isinstance(ts, six.string_types) else ts
            ttable = list(self._ttable) if not ts else [t for t in self._ttable if t > ts]
            return {
                'ttable': ttable,
                'confid': self._conf_id,
                'series': [{
                    'name': e.rule.name,
                    'task': e.rule.task,
                    'data': [
                        dict(se) for se in itertools.islice(e.stats, len(e.stats) - len(ttable), None)
                    ]
                } for e in sorted(six.itervalues(self._data), key=self.DataEntry.getOrder)],
            }


class RulezHistoryFetcher(BgTask):
    def __init__(self, cfg, log, db, history_tasks):
        super(RulezHistoryFetcher, self).__init__(log, cfg.period, 'Common_history_fetcher')

        self._cfg = cfg
        self._hi = db.hostinfo
        self._hi.ensure_index('skynet.updated')
        self._reloaded = 0
        self._tstag = None
        self._history_tasks = history_tasks

    def _do(self, first=False, forced=False):
        stts = time.time()
        rebuild = self._reloaded + self._cfg.full_reload_every < stts or first
        safets = dt.datetime.now() - dt.timedelta(seconds=self._cfg.safe_delay)

        # Initialize history updating
        need_rebuild = []
        for task in self._history_tasks:
            if self._stopping:
                raise BgTaskStoppingException()

            if not task.start_update_data(first, rebuild):
                self.log.info('Rules for {} have changed.'.format(task.name))
                need_rebuild.append(task)

        count = 0

        # Update history
        count += self._update_history(first, rebuild, safets, need_rebuild)

        if need_rebuild:
            count += self._update_history(first, True, safets, need_rebuild)

        if self._tstag > safets:  # Correct timestamp tag after the full rebuild
            self._tstag = safets

        self.log.info('Historical data processed: %d records in %.3gs.', count, time.time() - stts)

    def _update_history(self, first, rebuild, safets, need_rebuild):
        stts = time.time()

        fields = {'_id': False, 'skynet': True, 'gosky': True, 'host': True, 'last_update': True}
        count = 0

        if rebuild:
            self.log.info('Perform %s data fetch.', 'full' if not first else 'initial')
            query = {'skynet.updated': {'$exists': True}}
            self._tstag = dt.datetime.min
            self._reloaded = stts
        else:
            self.log.info('Perform partial data fetch.')
            query = {'skynet.updated': {'$lt': safets, '$gt': self._tstag}}

        for doc in self._hi.find(query, fields):
            ctstag = doc['skynet']['updated']
            if ctstag and self._tstag < ctstag:
                self._tstag = ctstag

            for task in self._history_tasks:
                if self._stopping:
                    raise BgTaskStoppingException()

                if rebuild and (not need_rebuild or task in need_rebuild):
                    task.update_data(doc)
                elif not rebuild and task not in need_rebuild:
                    task.update_data(doc)

            count += 1

        # Complete history updating
        for task in self._history_tasks:
            if self._stopping:
                raise BgTaskStoppingException()

            if rebuild and (not need_rebuild or task in need_rebuild):
                task.end_update_data()
            elif not rebuild and task not in need_rebuild:
                task.end_update_data()

        self.log.info('Historical data fetched: %d records in %.3gs.', count, time.time() - stts)
        return count


class StateMon(BgTask):
    def __init__(self, cfg, log, db, event_fetcher):
        self.cfg = cfg
        self.log = log
        self._hs = db.hbs_state
        self._hl = db.locks
        self._hl.ensure_index('name', unique=True)

        self._last_state = {}
        self._state_map = {
            'ok': 'OK',
            'warning': 'WARN',
            'critical': 'CRIT'
        }
        self._event_fetcher = event_fetcher
        super(StateMon, self).__init__(log, 30)

    def _do(self, first=False, forced=False):
        import datetime
        import collections

        # get events here just to remove them from the list
        events = self._event_fetcher()

        if not self._lock():
            return

        min_uptime = None
        min_uptime_host = None
        stats = {'busy': 0.0, 'queue': 0, 'rate': 0}
        stats_by_plugin = {
            'busy': collections.defaultdict(float),
            'queue': collections.defaultdict(int),
            'rate': collections.defaultdict(int)
        }

        for state in self._hs.find():
            if 'uptime' in state:
                if min_uptime is None:
                    min_uptime = state['uptime']
                    min_uptime_host = state['host']
                else:
                    if state['uptime'] < min_uptime:
                        min_uptime = state['uptime']
                        min_uptime_host = state['host']

            if 'last_update' in state:
                diff = (datetime.datetime.now() - state['last_update']).total_seconds()
                if diff > 300:
                    self._send_ev(
                        'heartbeat-server-is-dead',
                        'last status %d seconds ago (> 300)' % (diff, ),
                        'warning',
                        host=state['host']
                    )
                else:
                    self._send_ev(
                        'heartbeat-server-is-dead',
                        'last status %d seconds ago (< 300)' % (diff, ),
                        'ok',
                        host=state['host']
                    )

            if 'plugins' in state:
                for plugin_name, plugin_state in six.iteritems(state['plugins']):
                    for stat_key in stats:
                        if stat_key in plugin_state:
                            stats[stat_key] += plugin_state[stat_key]
                            stats_by_plugin[stat_key][plugin_name] += plugin_state[stat_key]

        if min_uptime < self.cfg.uptime.warn:
            self._send_ev(
                'too-low-uptime',
                'heartbeat-server [%s] has too low minimal uptime (%d secs < %d secs)' % (
                    min_uptime_host,
                    min_uptime,
                    self.cfg.uptime.crit if min_uptime < self.cfg.uptime.crit else self.cfg.uptime.warn
                ),
                'critical' if min_uptime < self.cfg.uptime.crit else 'warning'
            )
        else:
            self._send_ev(
                'too-low-uptime',
                'heartbeat-servers have good minimal uptime (%d secs > %d secs)' % (
                    min_uptime,
                    self.cfg.uptime.warn
                ),
                'ok'
            )

        for key, stat in six.iteritems(stats):
            thresh = getattr(self.cfg, key)
            if stat > thresh.warn:
                self._send_ev(
                    'too-high-%s' % (key, ),
                    'heartbeat-server: has too high %s (%s > %d)' % (
                        key, '%0.2f' % (stat, ) if isinstance(stat, float) else repr(stat),
                        thresh.crit if stat > thresh.crit else thresh.warn
                    ) + '\n\nBy plugin (sorted):\n' +
                    '\n'.join([
                        '  %s: %s' % (
                            name,
                            '%0.2f' % (stat, ) if isinstance(stat, float) else repr(stat)
                        )
                        for (name, stat)
                        in sorted(six.iteritems(stats_by_plugin[key]), key=lambda _: _[1], reverse=True)
                    ]),
                    'critical' if stat > thresh.crit else 'warning'
                )
            else:
                self._send_ev(
                    'too-high-%s' % (key, ),
                    'heartbeat-server: has good %s (%s < %d)' % (
                        key, '%0.2f' % (stat, ) if isinstance(stat, float) else repr(stat),
                        thresh.warn
                    ),
                    'ok'
                )

        self._do_service_alerting(events)

    def _do_service_alerting(self, events):
        for ev, value in six.iteritems(events):
            self.log.info('{}: {} "{}"'.format(ev, value[1], value[0]))
            self._send_ev(
                ev,
                value[0],
                value[1]
            )

    def _lock(self):
        import socket
        hostname = socket.gethostname()

        try:
            self._hl.insert({'host': hostname, 'name': 'golem', 'ts': time.time()})
            self.log.debug('golem lock: acquired first time')
            return True
        except:
            data = self._hl.find({'name': 'golem'})[0]

            if data.get('host', None) == hostname:
                self._hl.update(
                    {'host': hostname, 'name': 'golem'},
                    {'host': hostname, 'name': 'golem', 'ts': time.time()}
                )
                self.log.debug('golem lock: held by us')
                return True
            else:
                self.log.debug(
                    'golem lock: held by %s for %ds',
                    data.get('host', None),
                    time.time() - data.get('ts', 0)
                )
                if time.time() - data.get('ts', 0) > 600:
                    self._hl.remove({'host': data.get('host', None), 'name': 'golem'})

            return False

    def _send_ev(self, name, desc, status, host=None):
        import grequests
        last_status, last_desc, last_state_ts = self._last_state.get((name, host), (None, '', 0))
        if last_status == status and last_desc == desc and time.time() - last_state_ts < 600:
            self.log.debug(
                'Bypassing juggler update: last update %d seconds ago with same state',
                time.time() - last_state_ts
            )
            return

        # send Juggler event
        juggler_state = {
            "source": "heartbeat.yandex-team.ru",
            "events": [
                {
                    "description": desc,
                    "host": 'heartbeat' if host is None else host,
                    "instance": "",
                    "service": name,
                    "status": self._state_map[status],
                }
            ]
        }
        response = grequests.post(self.cfg.juggler.url, data=json.dumps(juggler_state))
        response.send()
        if response.response.status_code == 200:
            self.log.debug('Successfully sent stats to juggler:')
            self.log.debug('\t%s' % juggler_state)
            self._last_state[(name, host)] = (status, desc, time.time())
        else:
            self.log.error('Unable to send events to juggler: %s', response.response.text)


class MemUsageHostGroup(object):
    RESOLVE_PERIOD = 1800  # once per half-hour

    def __init__(self, group_name, hosts, log):
        self.log = log
        self._group_name = group_name
        self._hosts = hosts
        self._resolved_hosts = []
        self._mem_info = collections.defaultdict(list)
        self._last_resolve = 0

    def append_info(self, host, mem_type, value):
        if host in self._resolved_hosts:
            self._mem_info[mem_type].append(value)

    def reset_info(self):
        self._mem_info = collections.defaultdict(list)
        if time.time() - self._last_resolve > self.RESOLVE_PERIOD:
            try:
                self._resolved_hosts = Resolver().resolveHosts(self._hosts)
            except Exception as ex:
                self.log.warn('Memory usage group: {} = ({}) failed to resolve ({})'.format(
                    self._group_name, self._hosts, ex))
            else:
                self.log.info('Memory usage group: {} = ({}) initialized'.format(
                    self._group_name, self._hosts))
                self._last_resolve = time.time()

    def get_metrics(self):
        for mem_type, values in six.iteritems(self._mem_info):
            sorted_values = sorted(values)
            for value_type, func in [
                ('max', lambda v: v[-1]),
                ('avg', lambda v: six.moves.reduce(lambda x, y: x + y, v) / len(v) if len(v) > 0 else 0.0),
                ('p95', lambda v: percentile(v, 0.95))
            ]:
                result = func(sorted_values)
                name = '.'.join([self._group_name, mem_type.replace('/', '_'), value_type])
                yield (name, result)


class SkynetChecker(BgTask):
    BOT_API_URL = 'http://bot.yandex-team.ru/api/view.php?name=view_oops_hardware&separator=;'
    REQUEST_TIMEOUT = 180  # loading whole bot inventory can take quite long
    HOST_INFO_UPDATE_TIMEOUT = 300
    BOT_UPDATE_TIMEOUT = 3600

    def __init__(self, db, log, cfg):
        super(SkynetChecker, self).__init__(log, self.HOST_INFO_UPDATE_TIMEOUT)
        self._bot_hosts = set()
        self._host_info = db.hostinfo
        self._restarts = db.restart_metrics
        self._cfg = cfg
        self._info = {}
        self._bot_last_update = 0
        self._mem_usage_groups = []

        self._info['skynet_hosts_num'] = 0
        self._info['clean_hosts_num'] = 0
        self._info['clean_hosts'] = set()

        # init host groups for memory usage calculations
        for g_name, g_hosts in six.iteritems(self._cfg.memory_usage):
            self._mem_usage_groups.append(MemUsageHostGroup(g_name, g_hosts, self.log))

    def _do(self, first=False, forced=False):
        start = time.time()
        if forced:
            self.log.info('Re-build forced.')
        else:
            self.log.debug('Re-build by the schedule.')

        self._bot_last_update += self.HOST_INFO_UPDATE_TIMEOUT
        if first or self._bot_last_update >= self.BOT_UPDATE_TIMEOUT:
            if self._update_bot_info():
                self._bot_last_update = 0

        all_skynet_hosts, skynet_os, skynet_memory = self._read_host_info()

        # skynet hosts known for bot inventory
        skynet_hosts = all_skynet_hosts & self._bot_hosts
        self._info['skynet_hosts_num'] = len(skynet_hosts)

        for group in self._mem_usage_groups:
            group.reset_info()

        # calculate kernel version metrics
        counter_os_linux_3_18 = 0
        counter_os_linux_3_10 = 0
        counter_os_others = 0
        for host in all_skynet_hosts:
            version = skynet_os.get(host, None)
            if version.startswith('3.18'):
                counter_os_linux_3_18 += 1
            elif version.startswith('3.10'):
                counter_os_linux_3_10 += 1
            else:
                counter_os_others += 1

            # update memory usage info
            memory = skynet_memory.get(host, None)
            if memory:
                for mem_type, value in six.iteritems(memory):
                    for group in self._mem_usage_groups:
                        group.append_info(host, mem_type, value)

        # calculate CQ* restart metrics
        cqueue_restarts, cqudp_restarts = self._read_restarts_info()

        # hosts without skynet yet
        clean_hosts = self._bot_hosts - skynet_hosts
        self._info['clean_hosts_num'] = len(clean_hosts)
        self._info['clean_hosts'] = clean_hosts

        # send metrics to graphite
        if self._cfg.graphite.host:
            with GraphiteConnector(self.log, self._cfg.graphite.host, self._cfg.graphite.port) as connector:
                hostname = socket.gethostname().replace('.', '_')
                connector.send_report('deploy.{}.skynet_installed'.format(hostname), self._info['skynet_hosts_num'])
                connector.send_report('deploy.{}.not_installed'.format(hostname), self._info['clean_hosts_num'])
                connector.send_report('kernel.{}.linux_3_18'.format(hostname), counter_os_linux_3_18)
                connector.send_report('kernel.{}.linux_3_10'.format(hostname), counter_os_linux_3_10)
                connector.send_report('kernel.{}.others'.format(hostname), counter_os_others)
                connector.send_report('kernel.{}.no_info'.format(hostname), self._info['clean_hosts_num'])
                connector.send_report('cqueue.{}.restarts'.format(hostname), cqueue_restarts)
                connector.send_report('cqudp.{}.restarts'.format(hostname), cqudp_restarts)
                for group in self._mem_usage_groups:
                    for metric_name, metric_value in group.get_metrics():
                        connector.send_report('memory.{}.{}'.format(hostname, metric_name), metric_value)

        self.log.info('Skynet: %d, Clean %d', len(skynet_hosts), len(clean_hosts))
        self.log.info('Linux 3.18: %d, Linux 3.10: %d, Others: %d, No info: %d',
                      counter_os_linux_3_18, counter_os_linux_3_10, counter_os_others, self._info['clean_hosts_num'])

        self.log.debug('New data re-built in %.4gs.' % (time.time() - start))

    def _update_bot_info(self):
        unnamed = 0
        not_operated = 0
        hosts = set()

        try:
            opener = six.moves.urllib.request.build_opener()
            url = self.BOT_API_URL
            self.log.debug('Requesting bot info via %r', url)
            req = opener.open(url, timeout=self.REQUEST_TIMEOUT)
        except six.moves.urllib_error.URLError as ex:
            self.log.warn('Unable to fetch bot info: %r', str(ex))
            return False

        data = six.ensure_text(req.read())
        lines = data.split('\n')

        for line in lines:
            items = line.split(';')
            if len(items) > 3:
                if items[1]:
                    if items[2] == 'OPERATION':
                        hosts.add(items[1])
                    else:
                        not_operated += 1
                else:
                    unnamed += 1

        if len(hosts) == 0:
            self.log.warn('BOT returned 0 hosts. Something went wrong, so we just ignore it')
            return False

        self.log.info('BOT info: hosts = %d, unnamed hosts = %d, not in operation hosts = %d',
                      len(hosts), unnamed, not_operated)
        self._bot_hosts = hosts
        return True

    def _read_host_info(self):
        res = self._host_info.find(
            {'last_update': {'$gt': dt.datetime.now() - dt.timedelta(hours=2)}},
            {'_id': 0, 'host': 1, 'os': 1, 'skynet.srvmngr.memory': 1}
        )
        alive = set()
        os = dict()
        memory = dict()
        for r in res:
            alive.add(r['host'])
            try:
                os[r['host']] = r['os']['version']
                memory[r['host']] = r['skynet']['srvmngr']['memory']
            except KeyError:
                os[r['host']] = 'Missing'
                memory[r['host']] = None

        return alive, os, memory

    def _read_restarts_info(self):
        res = self._restarts.find(
            {'time': {'$gt': str(dt.datetime.now() - dt.timedelta(hours=2))}},
            {'_id': 0, 'service': 1}
        )
        cqueue, cqudp = 0, 0
        for r in res:
            if r['service'] == 'cqueue':
                cqueue += 1
            elif r['service'] == 'cqudp':
                cqudp += 1

        self.log.info("Service restarts: %d cqueue, %d cqudp", cqueue, cqudp)
        return cqueue, cqudp

    @property
    def info(self):
        return self._info


class OopsResolverTask(BgTask):
    def __init__(self, log, cfg):
        super(OopsResolverTask, self).__init__(log, 300)
        self._cfg = cfg
        self._cache = {}

    def _do(self, first=False, forced=False):
        for group, selector in six.iteritems(self._cfg):
            start = time.time()
            try:
                hosts = Resolver().resolveHosts(selector)
            except Exception:
                self.log.exception("Selector %r resolve failed in %.4gs", selector, time.time() - start)
            else:
                self._cache[group] = hosts
                self.log.info("Selector %r resolved in %.4gs", selector, time.time() - start)

    def get(self, group):
        return self._cache.get(group, [])


class OopsAggregationTask(BgTask):
    SOLOMON_API_URL = "https://solomon.yandex.net/api/v2/push?project=oops&cluster=any&service=Modules"
    REQUEST_TIMEOUT = 30
    SENSORS_IN_CHUNK = 100

    def __init__(self, db, log, cfg, resolver):
        super(OopsAggregationTask, self).__init__(log, 300)
        self._cfg = cfg
        self._oopsstat = db.oopsstat
        self._resolver = resolver

    def _do(self, first=False, forced=False):
        try:
            self._aggregate()
        except Exception:
            self.log.exception("oops stats aggregation failed")

    def _aggregate(self):
        import grequests

        start = time.time()
        now = dt.datetime.now()
        alive_threshold = now - dt.timedelta(seconds=self._cfg.alive_threshold)
        module_alive_threshold = now - dt.timedelta(seconds=self._cfg.module_alive_threshold)
        kw = {'<OK>': 0, '<DEAD>': 0}
        constructor = lambda: defaultdict(int, **kw)
        groups = {
            group: defaultdict(constructor)
            for group in self._cfg.groups
        }
        groups['<ALL>'] = defaultdict(constructor)

        for doc in self._oopsstat.find(None, {'_id': False}):
            host = doc['host']
            updated = doc['report']['updated']
            if updated < alive_threshold:
                continue  # just skip dead hosts

            for module, data in six.iteritems(doc.get('module', {})):
                status = ('<DEAD>'
                          if data['generated'] < module_alive_threshold
                          else '<OK>'
                          if data['traceback'] is None
                          else 'TB #' + str(hash(data['traceback']))
                          )
                groups['<ALL>'][module][status] += 1
                for group in self._cfg.groups:
                    if host in self._resolver.get(group):
                        groups[group][module][status] += 1

        sensors = [
            {
                'labels': {"group": group, "module": module, "status": _status},
                "value": groups[group][module][_status],
            }
            for group in groups
            for module in groups[group]
            for _status in groups[group][module]
        ]
        chunks = int(math.ceil(len(sensors) / float(self.SENSORS_IN_CHUNK)))
        for chunk in six.moves.xrange(chunks):
            report = {
                "commonLabels": {
                    "host": "heartbeat",
                },
                "sensors": sensors[chunk * self.SENSORS_IN_CHUNK: (chunk + 1) * self.SENSORS_IN_CHUNK],
            }
            chunklen = len(report['sensors'])
            try:
                report = json.dumps(report)
                url = self.SOLOMON_API_URL
                self.log.debug("[%d/%d] requesting solomon %s with data: %r", chunk + 1, chunks, url, report)

                headers = {
                    'Content-Type': 'application/json',
                    'User-Agent': 'SkynetHeartbeat/1.0',
                    'Authorization': 'OAuth ' + self._cfg.solomon_oauth,
                }
                req = grequests.post(url, data=report, headers=headers, timeout=self.REQUEST_TIMEOUT, verify=False)
                req.send()
                if req.response.status_code == 200:
                    self.log.info('[%d/%d] sent %d reports in %.4gs',
                                  chunk + 1, chunks, chunklen, time.time() - start)
                else:
                    self.log.exception("[%d/%d] failed to push oops stats to solomon (%s): %s",
                                       chunk + 1, chunks, req.response.status_code, req.response.text)
            except Exception:
                self.log.exception("[%d/%d] failed to push oops stats to solomon", chunk + 1, chunks)
            start = time.time()


class IssAggregationTask(BgTask):
    def __init__(self, log, cfg, db):
        super(IssAggregationTask, self).__init__(log, cfg.update_interval)
        self._cfg = cfg
        self._db = db
        self._dc_cache = {}
        self._host_cache = defaultdict(set)
        self._groups = [next(six.iterkeys(x)) for x in cfg.groups]
        self._iss_versions = {}
        self._iss_top_versions = {x: [['No data', '0.0', 'black'], ['', '', ''], ['', '', '']] for x in self._groups}
        self._iss_totals = {}
        self._iss_features = defaultdict(OrderedDict)
        self._iss_feature_tickets = {}

        for group in self._groups:
            for feature in self._cfg.features:
                self._iss_features[group][next(six.iterkeys(feature))] = [0, '', 'No data']

        for feature in self._cfg.features:
            k, v = next(six.iteritems(feature))
            self._iss_feature_tickets[k] = v.get('ticket', '')

        self._features_query_fields = {'_id': False, 'host': True, 'last_update': True}
        self._features_query = {'$or': []}
        for feature in self._cfg.features:
            q = next(six.itervalues(feature)).get('check', None)
            if not q:
                self.log.error('Invalid check: %r', next(six.iterkeys(feature)))
                continue

            and_q = {'$and': []}
            for k, v in six.iteritems(q):
                self._features_query_fields[k] = True
                and_q['$and'].append({k: v})
            self._features_query['$or'].append(and_q)

        self.log.info('FIELDS: %r', self._features_query_fields)
        self.log.info('QUERY: %r', self._features_query)

    @staticmethod
    def check_feature(query, rec):
        if not query:
            return False

        for k, v in six.iteritems(query):
            path = k.split('.')
            subtree = rec
            for p in path:
                try:
                    subtree = subtree[p]
                except Exception:
                    return False
            if subtree != v:
                return False

        return True

    def _do(self, first=False, forced=False):
        start = time.time()
        self._resolve_groups()
        self._update_versions()
        self._update_features()
        self.log.info('ISS aggregation task completed in: {}'.format(time.time() - start))

    def _update_versions(self):
        minimal_date = dt.datetime.now() - dt.timedelta(seconds=self._cfg.alive_threshold)
        dc_versions = {}
        dc_totals = {}
        top_versions = defaultdict(list)
        for group in self._groups:
            for i in six.moves.xrange(3):
                top_versions[group].append(['', '0.0', 'black'])

        for group in self._groups:
            if self._stopping:
                raise BgTaskStoppingException()

            hosts = self.get_hosts(group)
            self.log.info('%s has %d hosts', group, len(hosts))
            aggregation = self._db.aggregate([
                {'$match': {'last_update': {'$gt': minimal_date}, 'host': {'$in': hosts}}},
                {'$group': {'_id': '$version', 'count': {'$sum': 1}}},
                {'$sort': {'count': 1}}
            ])

            ver2count = defaultdict(int)
            for doc in aggregation:
                if doc['_id']:
                    ver = doc['_id']
                    ver2count[ver] += doc['count']
                else:
                    ver2count['None'] += doc['count']

            versions = sorted(six.iterkeys(ver2count), reverse=True)
            dc_versions[group] = list(map(lambda v: (v, ver2count[v], ), versions))
            total = sum(six.itervalues(ver2count))
            dc_totals[group] = total

            # calculate TOP here
            top = list(ver2count.items())
            top.sort(key=lambda i: i[1], reverse=True)

            for i, t in enumerate(top[:3]):
                if float(t[1])/total * 100 < 1:
                    break
                percentage = float(t[1])/total * 100
                top_versions[group][i] = [t[0], '{:.2f}'.format(percentage), self._get_color(percentage)]

        self._iss_versions = dc_versions
        self._iss_totals = dc_totals
        self._iss_top_versions = top_versions
        self.log.info('iss_versions: %r ', dc_versions)
        self.log.info('iss_totals: %r', dc_totals)
        self.log.info('iss_top_versions: %r', top_versions)

    def _update_features(self):
        # initialize feature info
        features_info = defaultdict(OrderedDict)
        for group in self._groups:
            for feature in self._cfg.features:
                features_info[group][next(six.iterkeys(feature))] = [0]

        # prepare mongo request
        now = dt.datetime.now()
        alive_threshold = dt.timedelta(seconds=self._cfg.alive_threshold)
        minimal_date = now - alive_threshold

        # do mongo request
        start = time.time()
        for rec in self._db.find(self._features_query, self._features_query_fields):
            if self._stopping:
                raise BgTaskStoppingException()

            if now - rec.get('last_update') > alive_threshold:
                continue
            groups = self.get_groups(rec.get('host'))
            if not groups:
                continue
            for feature in self._cfg.features:
                if self.check_feature(next(six.itervalues(feature)).get('check', None), rec):
                    for group in groups:
                        features_info[group][next(six.iterkeys(feature))][0] += 1
        self.log.info('Reading ISS data from mongo took {}'.format(time.time() - start))

        # prepare feature info
        start = time.time()
        for group in self._groups:
            if self._stopping:
                raise BgTaskStoppingException()

            hosts = self.get_hosts(group)
            total_query = {
                'config.agent': {'$exists': True},
                'host': {'$in': hosts},
                'last_update': {'$gt': minimal_date}
            }
            total_count = self._db.find(total_query).count()

            for key, feature in six.iteritems(features_info[group]):
                if not total_count:
                    feature.append('N/A')
                    feature.append('black')
                    continue

                percentage = (feature[0] / float(total_count) * 100.0)
                feature.append('%.2f' % percentage)
                feature.append(self._get_color(percentage))
                if percentage > 100:
                    self.log.error('{}: {} {} / {}'.format(group, key, feature[0], total_count))
                else:
                    self.log.info('{}: {} {} / {}'.format(group, key, feature[0], total_count))

        self.log.info('Processing ISS data took {}'.format(time.time() - start))
        self._iss_features = features_info

    def _get_color(self, percentage):
        for low, high, color in self._cfg.colorification:
            if high >= percentage >= low:
                return color

        self.log.error('Invalid colorification config for {}!'.format(percentage))
        return 'black'

    def _resolve_groups(self):
        host_cache = defaultdict(set)
        dc_cache = {}

        for group in self._cfg.groups:
            if self._stopping:
                raise BgTaskStoppingException()

            for name, selector in six.iteritems(group):
                start = time.time()
                try:
                    hosts = Resolver().resolveHosts(selector)
                except Exception:
                    self.log.exception(
                        "Failed to resolve selector %r in %.4gs. Update nothing!",
                        selector,
                        time.time() - start
                    )
                    return
                else:
                    dc_cache[name] = list(hosts)
                    for host in hosts:
                        host_cache[host].add(name)
                    self.log.info("Selector %r resolved in %.4gs", selector, time.time() - start)

        self._dc_cache = dc_cache
        self._host_cache = host_cache

    def get_hosts(self, group):
        return self._dc_cache.get(group, [])

    def get_groups(self, host):
        return self._host_cache.get(host)

    def get_versions(self):
        return self._iss_versions

    def get_top_versions(self):
        return self._iss_top_versions

    def get_all_groups(self):
        return self._groups

    def get_totals(self):
        return self._iss_totals

    def get_features(self):
        return self._iss_features

    def get_feature_tickets(self):
        return self._iss_feature_tickets

    def get_feature_query(self, feature):
        for f in self._cfg.features:
            k, v = next(six.iteritems(f))
            if k == feature:
                return v.get('check', {})


class GraphiteConnector(object):
    SKYNET_GRAPHITE_PREFIX = 'skynet.test.'
    GRAPHITE_HOST = 'graphite-qe.yandex-team.ru'
    GRAPHITE_PORT = 2003

    def __init__(self, log, host=None, port=None):
        self.log = log
        self.sock = None
        self.host = host if host else self.GRAPHITE_HOST
        self.port = port if port else self.GRAPHITE_PORT

    def __enter__(self):
        # setup connection to graphite
        self.sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        server_address = (self.host, self.port)
        try:
            self.sock.connect(server_address)
            self.log.debug('Connected to %s', server_address)
        except socket.error as ex:
            self.log.warn('Failed to connect: %s', ex)
            self.sock = None
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if self.sock:
            # close connection to graphite
            self.sock.shutdown(socket.SHUT_RDWR)
            self.sock.close()

    def send_report(self, name, value):
        if self.sock:
            message = '{}{} {} {}\n'.format(self.SKYNET_GRAPHITE_PREFIX, name, value, time.time())
            self.log.debug('Message: %s', message[0:-1])
            self.sock.sendall(message)


def percentile(sorted_list_of_values, percent):
    """
    Find the percentile of a list of values.

    @parameter sorted_list_of_values - is a list of values. Note N MUST BE already sorted.
    @parameter percent - a float value from 0.0 to 1.0.

    @return - the percentile of the values
    """
    if not sorted_list_of_values:
        return 0.0

    k = (len(sorted_list_of_values) - 1) * percent
    f = math.floor(k)
    c = math.ceil(k)
    if f == c:
        return sorted_list_of_values[int(k)]

    d0 = sorted_list_of_values[int(f)] * (c - k)
    d1 = sorted_list_of_values[int(c)] * (k - f)
    return d0 + d1
