"""
The purpose of this module is to make all needed stuff for starting daemon.
This includes initialization of our Context with DaemonContext inside.

During context initialization we will close file descriptors, setup
signal handlers and so on.

It is important what before context initialization gevent will NOT be imported.
"""

from __future__ import absolute_import, print_function, division

import argparse
import os
import sys

try:
    import gevent.coros  # noqa
except ImportError:
    # backward compatibility for external libraries using old gevent api
    import gevent.lock
    gevent.coros = gevent.lock
    sys.modules['gevent.coros'] = gevent.lock


from kernel.util.errors import formatException

from .context import Context


def parse_args():
    parser = argparse.ArgumentParser(
        formatter_class=lambda *args, **kwargs: argparse.ArgumentDefaultsHelpFormatter(*args, width=120, **kwargs),
        description='Heartbeat daemon.',
    )
    subparsers = parser.add_subparsers(title='ACTIONS', description='Actions to perform', dest='mode')
    daemon_parser = subparsers.add_parser('daemon', help='Start heartbeat daemon')
    daemon_parser.add_argument('-d', '--daemonize', action='store_true')
    daemon_parser.add_argument('-b', '--background', action='store_true')
    daemon_parser.add_argument(
        '-v', '--verbose', metavar='N', nargs='?', const=1, type=int, default=0,
        help='Increase verbosity by N times.'
    )
    daemon_parser.add_argument(
        '-C', action='append', metavar='KEY=VALUE', default=[],
        help='Override config (can be specified several times)'
    )
    daemon_parser.add_argument('--trace-config', action='store_true', help='Trace configuration to stdout')

    check_parser = subparsers.add_parser('check', help='Check if server is working')
    check_parser.add_argument('namespace', nargs='?', default='skynet', help='Service namespace')

    stop_parser = subparsers.add_parser('stop', help='Stop server')
    stop_parser.add_argument('namespace', nargs='?', default='skynet', help='Service namespace')

    return parser.parse_args()


def daemon(args):
    # Initialize daemon context
    try:
        ctx = Context()
        ctx.initialize(args)
    except SystemExit:
        raise
    except BaseException:
        print(
            'ERROR (initializing): {0}'.format(formatException()),
            file=sys.stderr
        )
        os._exit(1)  # pylint: disable=W0212

    try:
        with ctx:
            from ..main import main as realMain
            return realMain(ctx)
    except SystemExit:
        raise
    except BaseException:
        ctx.log.critical('Unhandled error: %s' % (formatException(), ))
        os._exit(1)  # pylint: disable=W0212


def main():
    args = parse_args()
    if args.mode == 'stop':
        from api.skycore import ServiceManager
        server = ServiceManager().get_service_python_api(args.namespace, 'heartbeat')
        server.call('stop')

    elif args.mode == 'check':
        from api.skycore import ServiceManager
        server = ServiceManager().get_service_python_api(args.namespace, 'heartbeat')
        res = server.call('ping')
        if not res:
            return 2

    else:
        return daemon(args)


if __name__ == '__main__':
    raise SystemExit(main())
