import re
import time
import locale
import socket
import sys
import inspect
import datetime as dt
from itertools import zip_longest
from collections import defaultdict, OrderedDict

import flask
import pymongo
import gevent
import gevent.pywsgi
try:
    import gevent.coros as coros
except ImportError:
    import gevent.lock as coros
import six

from kernel import util
from library.format import formatHosts
from distutils.version import LooseVersion

from . import bgtask


class TypeSafeLooseVersion(LooseVersion):
    @staticmethod
    def _cmp_item(a, b):
        if a < b:
            return -1
        elif a > b:
            return 1
        else:
            return 0

    def _cmp(self, other):
        try:
            return super()._cmp(other)
        except TypeError:
            for s, o in zip_longest(self.version, other.version):
                res = 0
                if s is None:
                    return -1
                elif o is None:
                    return 1
                elif isinstance(s, str) and isinstance(o, str):
                    res = self._cmp_item(s, o)
                elif isinstance(s, int) and isinstance(o, int):
                    res = self._cmp_item(s, o)
                elif isinstance(s, int) and isinstance(o, str):
                    return -1
                elif isinstance(s, str) and isinstance(o, int):
                    return 1

                if res != 0:
                    return res

            return res


class WebTab(object):
    title = ''

    def __init__(self, web, title, page_name):
        self.web = web
        self.title = title
        self.page_name = page_name

    def get_bgtasks(self):
        return []

    def get_pages(self):
        return []

    def __str__(self):
        return self.title


class WebApi(object):
    name = ''

    def __init__(self, web, name):
        self.web = web
        self.name = name

    def get_endpoints(self):
        return []

    def __str__(self):
        return self.name


class HiddenMainTab(WebTab):
    title = ''

    def __init__(self, web):
        super(HiddenMainTab, self).__init__(web, HiddenMainTab.title, '')

    def get_bgtasks(self):
        if self.web.ctx.cfg.get('golem'):
            return [
                ('statemon', bgtask.StateMon(
                    self.web.ctx.cfg.golem,
                    self.web.ctx.log,
                    self.web._primary_db,
                    self.web.get_events)),
                ('watchdog', bgtask.WatchdogTask(self.web.ctx.log))
            ]
        else:
            return []

    def get_pages(self):
        return [
            ('/', self.web.webRedirectToMainPage),
            ('/dashboard', self.web.webRedirectToMainPage),
            ('/ping', self.web.webPing)
        ]


class LacmusTab(WebTab):
    title = 'Lacmus'

    class LacmusService(object):
        def __init__(self, service_name, version_path, genisys_path, new_version_path):
            self.service_name = service_name
            self.version_path = version_path
            self.genisys_path = genisys_path
            self.new_version_path = new_version_path
            self.bgtask_name = self.service_name + '_rulez'

    def __init__(self, web):
        super(LacmusTab, self).__init__(web, LacmusTab.title, 'web_lacmus')
        self._services = OrderedDict()
        for service in self.web.cfg['lacmus']:
            for name, info in six.iteritems(service):
                info = info.split()
                self._services[name] = LacmusTab.LacmusService(
                    name,
                    info[0].split('.'),
                    info[1],
                    info[2].split('.') if len(info) > 2 else None
                )

    def get_bgtasks(self):
        tasks = []
        history_tasks = []
        for info in self._services.values():
            rulez_fetcher = bgtask.RulezFetcher(
                self.web.ctx.log,
                'http://api.genisys.yandex-team.ru/v1/rules?path={}&fmt=msgpack'.format(info.genisys_path),
                info.service_name
            )
            rulez_history = bgtask.RulezHistory(
                self.web.ctx.cfg,
                self.web.ctx.log,
                self.web.ctx.cfg.WorkdirPath,
                rulez_fetcher,
                info.version_path,
                info.new_version_path,
                info.service_name,
                info.version_path[-1],
                self.web.add_new_event
            )
            tasks.append((info.service_name + '_rulez_fetcher', rulez_fetcher))
            tasks.append((info.bgtask_name, rulez_history))
            history_tasks.append(rulez_history)

        rulez_history_fetcher = bgtask.RulezHistoryFetcher(
            self.web.ctx.cfg.lacmus.history,
            self.web.ctx.log,
            self.web._db,
            history_tasks
        )
        tasks.append(('Common_history_fetcher', rulez_history_fetcher))

        return tasks

    def get_pages(self):
        pages = [
            ('/lacmus', self.web_lacmus),
            ('/lacmus/<service>', self.web_lacmus),
            ('/versions', self.web.webVersions),
            ('/versions/<version>', self.web.webVersions),
            ('/hostinfo/<host>', self.web.getHostInfo),
            ('/outdated_report/<report_type>', self.web.webOutdatedReport)
        ]
        return pages

    def web_lacmus(self, service=None):
        if not service:
            services = [(svc, '/lacmus/' + svc) for svc in self._services.keys()]
            return self.web.webLacmusBase(services)

        return self.web.webLacmus(service, self._services[service].bgtask_name)


class ClusterHealthTab(WebTab):
    title = 'Cluster Health'

    def __init__(self, web):
        super(ClusterHealthTab, self).__init__(web, ClusterHealthTab.title, 'webHealth')

    def get_bgtasks(self):
        try:
            aux_tasks_config = self.web.ctx.cfg.tasks
            skynet_checker = bgtask.SkynetChecker(self.web._db, self.web.ctx.log, aux_tasks_config)
        except AttributeError:
            skynet_checker = None

        return [
            ('skynetChecker', skynet_checker)
        ]

    def get_pages(self):
        return [
            ('/health', self.web.webHealth),
            ('/versions', self.web.webVersions),
            ('/versions/<version>', self.web.webVersions),
            ('/versions/hostinfo/<host>', self.web.getHostInfo),
            ('/chart/<chartType>', self.web.webChart),
            ('/chart/<chartType>/<chartSubType>', self.web.webChart),
            ('/outdated_report/<report_type>', self.web.webOutdatedReport)
        ]


class GoSkyTab(WebTab):
    title = 'GoSky'

    def __init__(self, web):
        super(GoSkyTab, self).__init__(web, GoSkyTab.title, 'webGoSky')

    def get_pages(self):
        return [
            ('/gosky', self.web.webGoSky),
            ('/gosky/<version>', self.web.webGoSky),
            ('/outdated_report/<report_type>', self.web.webOutdatedReport)
        ]


class KernelTab(WebTab):
    title = 'Kernel'

    def __init__(self, web):
        super(KernelTab, self).__init__(web, KernelTab.title, 'webKernel')

    def get_pages(self):
        return [
            ('/kernel', self.web.webKernel),
            ('/kernel/<version>', self.web.webKernel)
        ]


class OSVersionTab(WebTab):
    title = 'OS'

    def __init__(self, web):
        super(OSVersionTab, self).__init__(web, OSVersionTab.title, 'webOsVersion')

    def get_pages(self):
        return [
            ('/os_version', self.web.webOsVersion),
            ('/os_version/<version>', self.web.webOsVersion)
        ]


class ISSVersionTab(WebTab):
    title = 'ISS'

    def __init__(self, web):
        super(ISSVersionTab, self).__init__(web, ISSVersionTab.title, 'webIssVersion')

    def get_pages(self):
        return [
            ('/iss', self.web.webIssVersion),
            ('/iss/<version>', self.web.webIssVersion)
        ]

    def get_bgtasks(self):
        if self.web.ctx.cfg.get('iss'):
            task = bgtask.IssAggregationTask(self.web.ctx.log, self.web.ctx.cfg.iss, self.web._db.issinfo)
            return [
                ('iss_aggregation', task)
            ]
        return []


class ISSFeatureTab(WebTab):
    title = 'ISS Features'

    def __init__(self, web):
        super(ISSFeatureTab, self).__init__(web, ISSFeatureTab.title, 'webIssFeature')

    def get_pages(self):
        return [
            ('/iss_features', self.web.webIssFeature),
            ('/iss_features/<feature>/<group>', self.web.webIssFeature)
        ]

    def get_bgtasks(self):
        if self.web.ctx.cfg.get('iss'):
            task = bgtask.IssAggregationTask(self.web.ctx.log, self.web.ctx.cfg.iss, self.web._db.issinfo)
            return [
                ('iss_aggregation', task)
            ]
        return []


class StatusTab(WebTab):
    title = 'Status'

    def __init__(self, web):
        super(StatusTab, self).__init__(web, StatusTab.title, 'webHBSState')

    def get_pages(self):
        return [
            ('/hbs_state', self.web.webHBSState)
        ]


class TimeSkewTab(WebTab):
    title = 'Time Skew'

    def __init__(self, web):
        super(TimeSkewTab, self).__init__(web, TimeSkewTab.title, 'webTimeSkew')

    def get_pages(self):
        return [
            ('/time_skew', self.web.webTimeSkew)
        ]


class PluginErrorsTab(WebTab):
    title = 'Plugin Errors'

    def __init__(self, web):
        super(PluginErrorsTab, self).__init__(web, PluginErrorsTab.title, 'webPluginErrors')

    def get_pages(self):
        return [
            ('/plugin_errors', self.web.webPluginErrors)
        ]


class OopsTab(WebTab):
    title = 'Oops'

    def __init__(self, web):
        super(OopsTab, self).__init__(web, self.title, 'web_oops_stats')

    def web_oops_stats(self, selector=None):
        collection = self.web._db['oopsstat']
        now = dt.datetime.now()
        alive_threshold = now - dt.timedelta(seconds=self.web._alive_threshold)
        module_alive_threshold = now - dt.timedelta(seconds=1200)
        if selector is None:
            hosts = None
        else:
            from library.sky.hostresolver import Resolver
            hosts = Resolver().resolveHosts(selector)

        dead = set()
        modules = defaultdict(lambda: defaultdict(set))
        okmodules = defaultdict(set)
        deadmodules = defaultdict(set)
        total = 0

        for doc in collection.find(None, {'_id': False}):
            host = doc['host']
            if hosts is not None and host not in hosts:
                continue
            total += 1
            updated = doc['report']['updated']
            if updated < alive_threshold:
                dead.add(host)
                continue
            for module, data in six.iteritems(doc.get('module', {})):
                modules[module].get('')  # ensure dict creation
                if data['generated'] < module_alive_threshold:
                    deadmodules[module].add(host)
                elif data['traceback'] is None:
                    okmodules[module].add(host)
                else:
                    modules[module][str(data['traceback'])].add(host)

        for mod in modules:
            okmodules[mod] = (len(okmodules[mod]),
                              formatHosts(okmodules[mod], yrFormat=False, addDomain=True))
            deadmodules[mod] = (len(deadmodules[mod]),
                                formatHosts(deadmodules[mod], yrFormat=False, addDomain=True))
            for tb in modules[mod]:
                modules[mod][tb] = (len(modules[mod][tb]),
                                    formatHosts(modules[mod][tb], yrFormat=False, addDomain=True))

        return flask.render_template(
            'oops_info.html',
            total=total,
            dead=dead,
            selector=selector,
            modules=modules,
            okmodules=okmodules,
            deadmodules=deadmodules,
        )

    def web_host_info(self, host):
        collection = self.web._db['oopsstat']
        now = dt.datetime.now()
        alive_threshold = now - dt.timedelta(seconds=self.web._alive_threshold)
        module_alive_threshold = now - dt.timedelta(seconds=1200)

        docs = list(collection.find({'host': host}, {'_id': False}))
        if not docs:
            return flask.render_template(
                'oops_info_host.html',
                host=host,
                found=False,
            )

        doc = docs[0]
        updated = doc['report']['updated']
        is_dead = updated < alive_threshold
        modules = {}
        for module, data in six.iteritems(doc.get('module', {})):
            status = data['traceback'] or ''
            color = ('red'
                     if data['generated'] < module_alive_threshold
                     else 'green'
                     if data['traceback'] is None
                     else 'orange'
                     )
            modules[module] = (data['generated'], status, color)

        return flask.render_template(
            'oops_info_host.html',
            host=host,
            found=True,
            is_dead=is_dead,
            last_update=updated,
            modules=modules,
        )

    def get_pages(self):
        return [
            ('/oops_info', self.web_oops_stats),
            ('/oops_info/group/<selector>', self.web_oops_stats),
            ('/oops_info/host/<host>', self.web_host_info),
        ]

    def get_bgtasks(self):
        if self.web.ctx.cfg.get('oops'):
            resolver = bgtask.OopsResolverTask(self.web.ctx.log, self.web.ctx.cfg.oops.groups)
            aggregator = bgtask.OopsAggregationTask(self.web._db,
                                                    self.web.ctx.log,
                                                    self.web.ctx.cfg.oops,
                                                    resolver)
            return [
                ('oops_resolver', resolver),
                ('oops_aggregator', aggregator),
            ]
        return []


class VmUsageStatsApi(WebApi):
    name = 'vm_usage_stats'

    def __init__(self, web):
        super(VmUsageStatsApi, self).__init__(web, self.name)

    def _prepare_mongo_document(self, doc):
        if not doc:
            return None
        if '_id' in doc:
            del doc['_id']
        if isinstance(doc.get('generated'), dt.datetime):
            doc['generated'] = doc['generated'].isoformat()
        if isinstance(doc.get('last_update'), dt.datetime):
            doc['last_update'] = doc['last_update'].isoformat()
        if isinstance(doc.get('received'), dt.datetime):
            doc['received'] = doc['received'].isoformat()
        if isinstance(doc.get('initialised'), dt.datetime):
            doc['initialised'] = doc['initialised'].isoformat()
        return doc

    def _get_report_data(self, collection, vm_addr):
        coll = self.web._db[collection]
        host_info = coll.find_one({'host': vm_addr})
        return self._prepare_mongo_document(host_info)

    def vm_usage_stats_json(self, fqdn):
        vm_usage_data = {
            'who': self._get_report_data('who', fqdn),
            'last': self._get_report_data('last', fqdn),
            'dutop': self._get_report_data('dutop', fqdn),
        }
        if not any(vm_usage_data.values()):
            flask.abort(404)
        return flask.jsonify(vm_usage_data)

    def get_endpoints(self):
        return [
            ('/vm_usage_stats_json/<fqdn>', self.vm_usage_stats_json),
        ]


class WebTabManager(object):
    def __init__(self, web):
        # TODO: auto-detect WebTabs
        known_tabs = OrderedDict()
        classes = inspect.getmembers(sys.modules[__name__], inspect.isclass)
        for cls_name, cls in classes:
            if issubclass(cls, WebTab) and cls.title:
                known_tabs[cls.title] = cls(web)

        self._web = web
        self.log = web.ctx.log.getChild('tab_mgr')

        # get enabled tabs from web config
        enabled_tabs = self._web.cfg.get('enabled_tabs')

        if not enabled_tabs:
            # if it's not found, all tabs will be enabled
            self.log.info('Enabled tabs is empty - all known tabs will be enabled')
            self._tabs = list(known_tabs.values())
        else:
            self.log.info('Enabled tabs: %s', enabled_tabs)
            self._tabs = [known_tabs[tab] for tab in enabled_tabs if tab in known_tabs]

        self.log.info('Activated tabs: %s', [str(tab) for tab in self._tabs])
        self._tabs.append(HiddenMainTab(web))

    def prepare_bgtasks(self):
        bgtasks = {}
        for tab in self._tabs:
            tasks = tab.get_bgtasks()
            for (name, task) in tasks:
                if name not in bgtasks:
                    bgtasks[name] = task

        return bgtasks

    def prepare_pages(self):
        pages = {}
        for tab in self._tabs:
            self.log.info('Preparing tab: %s', str(tab))
            tab_pages = tab.get_pages()
            for (name, func) in tab_pages:
                if name not in pages:
                    pages[name] = func

        return pages

    def prepare_navbar(self):
        return [(tab.title, tab.page_name) for tab in self._tabs if tab.title]


class WebApiManager(object):
    def __init__(self, web):
        # TODO: auto-detect WebApis
        known_apis = OrderedDict()
        classes = inspect.getmembers(sys.modules[__name__], inspect.isclass)
        for cls_name, cls in classes:
            if issubclass(cls, WebApi) and cls.name:
                known_apis[cls.name] = cls(web)

        self._web = web
        self.log = web.ctx.log.getChild('api_mgr')

        # get enabled tabs from web config
        enabled_apis = self._web.cfg.get('enabled_apis')

        if not enabled_apis:
            # if it's not found, all apis will be enabled
            self.log.info('Enabled apis is empty - all known apis will be enabled')
            self._apis = list(known_apis.values())
        else:
            self.log.info('Enabled apis: %s', enabled_apis)
            self._apis = [known_apis[tab] for tab in enabled_apis if tab in known_apis]

        self.log.info('Activated apis: %s', [str(api) for api in self._apis])

    def prepare_endpoints(self):
        endpoints = {}
        for api in self._apis:
            self.log.info('Preparing endpoints: %s', str(api))
            api_endpoints = api.get_endpoints()
            for (name, func) in api_endpoints:
                if name not in endpoints:
                    endpoints[name] = func

        return endpoints


class Web(object):
    # Initialization {{{
    def __init__(self, ctx, appstopper, db, primary_db):
        self.ctx = ctx
        self.cfg = ctx.cfg.web
        self._alive_threshold = ctx.cfg.lacmus.history.alive_threshold
        self.log = ctx.log.getChild('web')
        self.log.debug('Initializing')

        self._db = db
        self._primary_db = primary_db
        self._wsgi = None
        self._tab_manager = WebTabManager(self)
        self._api_manager = WebApiManager(self)

        bgtasks = self._tab_manager.prepare_bgtasks()
        self._bgp = bgtask.Pool(ctx.log, appstopper, **bgtasks)
        self._events = dict()
        self._eventsLock = coros.RLock()

        try:
            locale.setlocale(locale.LC_ALL, 'ru_RU.UTF-8')
        except:
            locale.setlocale(locale.LC_ALL, 'en_US.UTF-8')

    # Initialization }}}

    # Management (start, stop, join) {{{
    def start(self):
        assert self._wsgi is None

        self._flask_app = flask.Flask(
            __name__,
            instance_path=self.cfg.flask_app.instance_path,
            template_folder=self.cfg.flask_app.template_folder,
            static_folder=self.cfg.flask_app.static_folder,
            static_url_path=self.cfg.flask_app.static_url_path
        )
        self._flask_app.config.update(self.cfg.flask_config)

        pages = self._tab_manager.prepare_pages()
        for url, func in six.iteritems(pages):
            self._flask_app.add_url_rule(url, view_func=func)

        self._flask_app.context_processor(self._navbarCtxProc)

        endpoints = self._api_manager.prepare_endpoints()
        for url, func in six.iteritems(endpoints):
            self._flask_app.add_url_rule(url, view_func=func)

        self._wsgi_sock = gevent.socket.socket(gevent.socket.AF_INET6)
        self._wsgi_sock.setsockopt(gevent.socket.SOL_SOCKET, gevent.socket.SO_REUSEADDR, 1)
        self._wsgi_sock.setsockopt(gevent.socket.IPPROTO_IPV6, gevent.socket.IPV6_V6ONLY, 0)
        self._wsgi_sock.bind((self.cfg.bind_host, self.cfg.bind_port))
        self._wsgi_sock.listen(self.cfg.backlog)

        self._wsgi = gevent.pywsgi.WSGIServer(
            self._wsgi_sock,
            application=self._flask_app,
            log=None
        )
        self._wsgi.start()

        self._bgp.start()
        self.log.info('Listening for HTTP at %r:%d', self.cfg.bind_host, self.cfg.bind_port)
        return self

    def stop(self):
        assert self._wsgi is not None
        self._wsgi.stop(timeout=self.cfg.stop_timeout)
        self._wsgi_sock.close()
        self.log.info('Stopped HTTP server')
        self._bgp.stop()
        return self

    def ping(self):
        return self._bgp.ping()

    def join(self):
        self._wsgi._stop_event.wait()
        self._bgp.join()
    # Management (start, stop, join) }}}

    # Jinja helpers {{{
    def _jinja_version_sort(self, lst, reverse=False):
        return sorted(lst, key=TypeSafeLooseVersion, reverse=reverse)
    # Jinja helpers }}}

    # Nav bars {{{
    def _navbarCtxProc(self):
        ep = flask.request.endpoint

        navbar = []

        for title, page in self._tab_manager.prepare_navbar():
            try:
                url = flask.url_for(page)
            except Exception:
                self.log.warning('Cant find url for page %r' % (page, ))
                url = page

            navbar.append({
                'url': url,
                'page': page,
                'title': title,
                'active': True if ep.startswith(page) else False
            })

        return {
            'navbar': navbar,
            'brand': 'Skynet HeartBeat',
            'hostname': socket.gethostname(),
            'numfmt': lambda fmt, x: six.ensure_text(locale.format(fmt, x, grouping=True), 'utf-8'),
            'tdfmt': util.td2str,
            'dtfmt': util.dt2str,
        }
    # Nav bars }}}

    # Charts {{{
    def _getSkynetCurrentVersions(self):
        aliveThreshold = dt.datetime.now() - dt.timedelta(seconds=self._alive_threshold)
        aggregation = self._db.hostinfo.aggregate([
            {'$project': {
                'alive': {'$cond': [{'$gt': ['$last_update', aliveThreshold]}, 1, 0]},
                'version': '$skynet.version',
            }},
            {'$group': {
                '_id': '$version',
                'total': {'$sum': 1},
                'alive': {'$sum': '$alive'}
            }}
        ])

        ver2count = {}
        # add non-skynet hosts
        if self._bgp.skynetChecker:
            skynet_host_info = self._bgp.skynetChecker.info
            if skynet_host_info['clean_hosts_num'] > 0:
                ver2count['no_skynet'] = ('no_skynet', 0, skynet_host_info['clean_hosts_num'], )

        for doc in filter(lambda x: x['_id'], aggregation):
            alive = doc['alive']
            ver2count[doc['_id']] = (doc['_id'], alive, doc['total'] - alive, )

        versions = self._jinja_version_sort(ver2count.keys(), reverse=True)
        return versions, [ver2count[v][:2] for v in versions], [ver2count[v][::2] for v in versions]

    def _chartSkynetCurrentVersions(self):
        vers, alive, dead = self._getSkynetCurrentVersions()
        return {
            'version': 1,
            'series': [
                {
                    'name': 'Alive',
                    'data': alive,
                },
                {
                    'visible': False,
                    'name': 'Dead',
                    'data': dead,
                },
            ],
            'xAxis': {'categories': vers}
        }
    # Charts }}}

    @staticmethod
    def _colorer(x, negative=False):
        x = 100 - x if negative else x
        return \
            'EE%02X00' % max(int(255 / 50.0 * x), 0) \
            if x < 50 else \
            '%02XEE00' % max(int(255 - 255 / 50.0 * (x - 50)), 0)

    @staticmethod
    def _shortify(host):
        # do not shortify anymore
        # return host.replace('.yandex.ru', '')
        return host

    def _rulezSelector(self, source):
        hosts = set()
        req = flask.request
        rulez = set(str(x) for x in req.args.getlist('rule'))
        types = set(str(x) for x in req.args.getlist('type'))
        storage = self._bgp[source].storage

        for r in rulez:
            for t in types:
                hosts |= storage[r].getattrbyname(t)
        rulez = (
            [(x, x in rulez) for x in storage.keys()],
            [(x, x in types) for x in bgtask.RulezHistory.StatsEntry.names()]
        )
        return rulez, hosts

    def _webDeadlines(self, defid=0, prefix='Show data for last '):
        td = dt.timedelta
        td2s = lambda x: x.seconds + x.days * 86400
        arg = flask.request.args.get('deadline', None)
        curr = (td(seconds=int(arg)) if arg != 'forever' else td.min) if arg else None
        OPTIONS = \
            [td(minutes=30)] + \
            [td(hours=i) for i in [1, 3, 6, 12]] + \
            [td(days=i) for i in [1, 3, 7, 14, 30]] + [td.min]
        lst = [
            (
                td2s(t) if t != td.min else 'forever',
                (
                    ('%s%s' % (prefix, util.td2str(t, True)))
                    if t != td.min
                    else 'Show all collected data'
                ),
                t == curr if curr else i == defid,
            )
            for i, t in enumerate(OPTIONS)
        ]
        return OPTIONS[defid] if not curr else curr, lst

    def webRedirectToMainPage(self):
        page_name = self._tab_manager._tabs[0].page_name
        return flask.redirect(flask.url_for(page_name))

    def webChart(self, chartType, chartSubType=None):
        meth = getattr(
            self,
            '_chart%s%s' % (
                chartType.replace('-', ' ').title().replace(' ', ''),
                (
                    chartSubType.replace('-', ' ').title().replace(' ', '') if chartSubType else ''
                ),
            ), None
        )

        if meth is None:
            self.log.warning('Invalid chart: type=%r, subtype=%r' % (chartType, chartSubType))
            flask.abort(404)

        resp = flask.make_response(flask.json.dumps(meth()))
        resp.headers['Content-Type'] = 'application/json'
        return resp

    def webPing(self):
        resp = flask.make_response('pong')
        resp.headers['Content-Type'] = 'text/plain'
        return resp

    def webHealth(self):
        vers, alive, dead = self._getSkynetCurrentVersions()
        stats = [sum(x[1] for x in c) for c in [alive, dead]]
        return flask.render_template(
            'dashboard.html',
            colorer=self._colorer,
            stats=[sum(stats)] + stats,
            skynetVersions=[(v, alive[i][0], alive[i][1], dead[i][1], ) for (i, v) in enumerate(vers)],
        )

    def webLacmus(self, service, bg_task):
        req = flask.request
        if req.args.get('json') is None and req.accept_mimetypes.accept_html:
            return flask.render_template('lacmus.html', service=service, source=bg_task, keepLoading=True)

        resp = flask.make_response(flask.json.dumps(self._bgp[bg_task].data(req.args.get('ts'))))
        resp.headers['Content-Type'] = 'application/json'
        return resp

    def webLacmusBase(self, services):
        return flask.render_template('lacmus_all.html', services=services)

    def webGoSky(self, version=None):
        req = flask.request
        ver2str = lambda v: '.'.join(map(str, v)) if v else 'None'
        if version:
            if req.args.get('json') is None and req.accept_mimetypes.accept_html:
                return flask.render_template('gosky_versions.html', keepLoading=True)

            data = []
            now = dt.datetime.now()
            aliveThreshold = dt.timedelta(seconds=self._alive_threshold)
            query = (
                {'gosky.version': list(map(int, version.split('.')))}
                if version.lower() not in ('none', 'error', )
                else {'gosky.version': {'$exists': False}}
                if version.lower() != 'error'
                else {'gosky.error': {'$exists': True}}
            )

            hostinfo = self._db.hostinfo
            hostinfo.ensure_index('last_update')
            hostinfo.ensure_index('gosky.version')
            for doc in hostinfo.find(
                query,
                {'_id': False, 'host': True, 'last_update': True, 'gosky': True, 'skynet': True},
                sort=[('last_update', pymongo.DESCENDING, )]
            ):
                diff = now - doc['last_update']
                gosky_info = doc.get('gosky', {})
                data.append((
                    self._shortify(doc['host']),
                    '.'.join(str(d) for d in gosky_info.get('version', ['Missing'])),
                    util.dt2str(doc['last_update']),
                    util.td2str(diff),
                    gosky_info.get('error', None),
                    diff > aliveThreshold,
                ))

            resp = flask.make_response(flask.json.dumps(
                {'full_list': data, 'quick_list': formatHosts([d[0] for d in data], yrFormat=False, addDomain=True)}
            ))
            resp.headers['Content-Type'] = 'application/json'
            return resp

        minimal_date = dt.datetime.now() - dt.timedelta(seconds=self._alive_threshold)

        aggregation = self._db.hostinfo.aggregate([
            {'$match': {'last_update': {'$gt': minimal_date}}},
            {'$group': {'_id': '$gosky.version', 'count': {'$sum': 1}}},
            {'$sort': {'count': 1}}
        ])

        ver2count = {}
        for doc in aggregation:
            ver2count[ver2str(doc['_id'])] = doc['count']

        aggregation = self._db.hostinfo.aggregate([
            {'$match': {'last_update': {'$gt': minimal_date}}},
            {'$group': {'_id': '$gosky.error', 'count': {'$sum': 1}}},
        ])
        ver2count['Error'] = sum(r['count'] for r in aggregation if r['_id'])

        versions = self._jinja_version_sort(ver2count.keys(), reverse=True)
        return flask.render_template(
            'gosky.html',
            total=sum(ver2count.values()),
            versions=list(map(lambda v: (v, ver2count[v], ), versions)),
        )

    def webKernel(self, version=None):
        req = flask.request
        aliveThreshold = dt.timedelta(seconds=self._alive_threshold)
        minimal_date = dt.datetime.now() - aliveThreshold

        if version:
            if req.args.get('json') is None and req.accept_mimetypes.accept_html:
                return flask.render_template('kernel_versions.html', keepLoading=True)

            data = []
            now = dt.datetime.now()
            hostinfo = self._db.hostinfo
            hostinfo.ensure_index('last_update')
            hostinfo.ensure_index('os.version')
            if version != 'None':
                regex = re.compile('^' + version)
                for doc in hostinfo.find(
                        {'os.version': {'$regex': regex}},
                        {'_id': False, 'host': True, 'last_update': True, 'os': True},
                        sort=[('last_update', pymongo.DESCENDING, )]
                ):
                    diff = now - doc['last_update']
                    ver = doc.get('os')['version'].split('.')
                    if len(ver) >= 2 and diff < aliveThreshold:
                        data.append((
                            self._shortify(doc['host']),
                            '.'.join([ver[0], ver[1]]),
                            util.dt2str(doc['last_update']),
                            util.td2str(diff),
                            '{}: {}'.format(doc.get('os')['name'], '.'.join(ver)),
                            False,
                        ))
            else:
                for doc in hostinfo.find(
                        {'os.version': {'$exists': False}},
                        {'_id': False, 'host': True, 'last_update': True},
                        sort=[('last_update', pymongo.DESCENDING, )]
                ):
                    diff = now - doc['last_update']
                    if diff < aliveThreshold:
                        data.append((
                            self._shortify(doc['host']),
                            'None',
                            util.dt2str(doc['last_update']),
                            util.td2str(diff),
                            '',
                            False,
                        ))

            resp = flask.make_response(flask.json.dumps(
                {'full_list': data, 'quick_list': formatHosts([d[0] for d in data], yrFormat=False, addDomain=True)}
            ))
            resp.headers['Content-Type'] = 'application/json'
            return resp

        aggregation = self._db.hostinfo.aggregate([
            {'$match': {'last_update': {'$gt': minimal_date}}},
            {'$group': {'_id': '$os.version', 'count': {'$sum': 1}}},
            {'$sort': {'count': 1}}
        ])

        ver2count = defaultdict(int)
        for doc in aggregation:
            if doc['_id']:
                ver = doc['_id'].split('.')
                ver2count['.'.join([ver[0], ver[1]])] += doc['count']
            else:
                ver2count['None'] += doc['count']

        versions = sorted(ver2count.keys(), reverse=True)
        return flask.render_template(
            'kernel.html',
            total=sum(ver2count.values()),
            versions=list(map(lambda v: (v, ver2count[v], ), versions)),
        )

    def webOsVersion(self, version=None):
        req = flask.request
        aliveThreshold = dt.timedelta(seconds=self._alive_threshold)
        minimal_date = dt.datetime.now() - aliveThreshold

        if version:
            if req.args.get('json') is None and req.accept_mimetypes.accept_html:
                return flask.render_template('os_version_versions.html', keepLoading=True)

            data = []
            now = dt.datetime.now()
            hostinfo = self._db.hostinfo
            hostinfo.ensure_index('last_update')
            hostinfo.ensure_index('os.name')
            if version != 'None':
                os_ver = version.split(':')
                if len(os_ver) == 3:
                    for doc in hostinfo.find(
                            {
                                'os.name': os_ver[0],
                                'os.os_name': os_ver[1] if os_ver[1] != 'Unknown' else {'$exists': False},
                                'os.os_version': os_ver[2] if os_ver[2] != 'Unknown' else {'$exists': False}
                            },
                            {'_id': False, 'host': True, 'last_update': True},
                            sort=[('last_update', pymongo.DESCENDING, )]
                    ):
                        diff = now - doc['last_update']
                        if diff < aliveThreshold:
                            data.append((
                                self._shortify(doc['host']),
                                version,
                                util.dt2str(doc['last_update']),
                                util.td2str(diff),
                                '',
                                False,
                            ))
            else:
                for doc in hostinfo.find(
                        {'os.name': {'$exists': False}},
                        {'_id': False, 'host': True, 'last_update': True},
                        sort=[('last_update', pymongo.DESCENDING, )]
                ):
                    diff = now - doc['last_update']
                    if diff < aliveThreshold:
                        data.append((
                            self._shortify(doc['host']),
                            'None',
                            util.dt2str(doc['last_update']),
                            util.td2str(diff),
                            '',
                            False,
                        ))

            resp = flask.make_response(flask.json.dumps(
                {'full_list': data, 'quick_list': formatHosts([d[0] for d in data], yrFormat=False, addDomain=True)}
            ))
            resp.headers['Content-Type'] = 'application/json'
            return resp

        aggregation = self._db.hostinfo.aggregate([
            {'$match': {'last_update': {'$gt': minimal_date}}},
            {
                '$group':
                {
                    '_id': {
                        'os_name': '$os.name',
                        'os_distr': '$os.os_name',
                        'os_version': '$os.os_version',
                    },
                    'count': {'$sum': 1}
                }
            },
            {'$sort': {'count': 1}}
        ])

        ver2count = defaultdict(int)
        for doc in aggregation:
            if doc['_id']:
                ver_str = '{}:{}:{}'.format(doc['_id']['os_name'],
                                            doc['_id'].get('os_distr', 'Unknown'),
                                            doc['_id'].get('os_version', 'Unknown'))
                ver2count[ver_str] += doc['count']
            else:
                ver2count['None'] += doc['count']

        versions = sorted(ver2count.keys(), reverse=True)
        return flask.render_template(
            'os_version.html',
            total=sum(ver2count.values()),
            versions=list(map(lambda v: (v, ver2count[v], ), versions)),
        )

    def webIssVersion(self, version=None):
        req = flask.request

        if version:
            if req.args.get('json') is None and req.accept_mimetypes.accept_html:
                return flask.render_template('iss_versions.html', keepLoading=True)

            alive_threshold = dt.timedelta(seconds=self.ctx.cfg.iss.alive_threshold)
            dc = req.args.get('dc', [])
            hosts = self._bgp.iss_aggregation.get_hosts(dc) if dc else []
            now = dt.datetime.now()
            issinfo = self._db.issinfo
            issinfo.ensure_index('last_update')
            issinfo.ensure_index('version')

            data = []
            if version != 'None':
                regex = re.compile('^' + version)
                spec = {
                    'version': {'$regex': regex},
                    'host': {'$in': hosts}
                } if hosts else {
                    'version': {'$regex': regex}
                }

                for doc in issinfo.find(
                        spec,
                        {'_id': False, 'host': True, 'last_update': True, 'version': True},
                        sort=[('last_update', pymongo.DESCENDING, )]
                ):
                    diff = now - doc['last_update']
                    ver = doc['version']
                    if diff < alive_threshold:
                        data.append((
                            self._shortify(doc['host']),
                            ver,
                            util.dt2str(doc['last_update']),
                            util.td2str(diff),
                            '',
                            False,
                        ))
            else:
                spec = {
                    'version': {'$exists': False},
                    'host': {'$in': hosts}
                } if hosts else {
                    'version': {'$exists': False}
                }
                for doc in issinfo.find(
                        spec,
                        {'_id': False, 'host': True, 'last_update': True},
                        sort=[('last_update', pymongo.DESCENDING, )]
                ):
                    diff = now - doc['last_update']
                    if diff < alive_threshold:
                        data.append((
                            self._shortify(doc['host']),
                            'None',
                            util.dt2str(doc['last_update']),
                            util.td2str(diff),
                            '',
                            False,
                        ))

            resp = flask.make_response(flask.json.dumps(
                {'full_list': data, 'quick_list': formatHosts([d[0] for d in data], yrFormat=False, addDomain=True)}
            ))
            resp.headers['Content-Type'] = 'application/json'
            return resp

        versions = self._bgp.iss_aggregation.get_versions()
        totals = self._bgp.iss_aggregation.get_totals()
        groups = self._bgp.iss_aggregation.get_all_groups()
        top_versions = self._bgp.iss_aggregation.get_top_versions()

        return flask.render_template(
            'iss.html',
            groups=groups,
            total=totals,
            versions=versions,
            top_versions=top_versions
        )

    def webIssFeature(self, feature=None, group=None):
        req = flask.request

        if feature and group:
            if req.args.get('json') is None and req.accept_mimetypes.accept_html:
                return flask.render_template('iss_feature_details.html', keepLoading=True)

            now = dt.datetime.now()
            alive_threshold = dt.timedelta(seconds=self.ctx.cfg.iss.alive_threshold)
            minimal_date = now - alive_threshold
            data = []

            issinfo = self._db.issinfo
            hosts = self._bgp.iss_aggregation.get_hosts(group)
            feature_query = self._bgp.iss_aggregation.get_feature_query(feature)
            fields = {'_id': False, 'host': True, 'last_update': True}
            for k, v in feature_query.items():
                fields[k] = True

            for doc in issinfo.find(
                    {
                        'config.agent': {'$exists': True},
                        'host': {'$in': hosts},
                        'last_update': {'$gt': minimal_date}
                    },
                    fields,
                    sort=[('last_update', pymongo.DESCENDING, )]
            ):
                diff = now - doc['last_update']
                feature_enabled = self._bgp.iss_aggregation.check_feature(feature_query, doc)
                data.append((
                    self._shortify(doc['host']),
                    feature,
                    util.dt2str(doc['last_update']),
                    util.td2str(diff),
                    'ON' if feature_enabled else 'OFF',
                    not feature_enabled
                ))

            resp = flask.make_response(flask.json.dumps(
                {
                    'full_list': data,
                    'quick_on_list': formatHosts([d[0] for d in data if not d[5]], yrFormat=False, addDomain=True),
                    'quick_off_list': formatHosts([d[0] for d in data if d[5]], yrFormat=False, addDomain=True)
                }
            ))
            resp.headers['Content-Type'] = 'application/json'
            return resp

        groups = self._bgp.iss_aggregation.get_all_groups()
        features = self._bgp.iss_aggregation.get_features()
        feature_tickets = self._bgp.iss_aggregation.get_feature_tickets()

        return flask.render_template(
            'iss_features.html',
            groups=groups,
            features=features,
            feature_tickets=feature_tickets,
            columns=self.ctx.cfg.iss.columns_in_row
        )

    def webOutdatedReport(self, report_type):
        RESULT_LIMIT = 1000
        QUERY_HOSTS_LIMIT = 1000

        rulez = None
        hosts = set()
        req = flask.request
        log = self.log.getChild('report')

        gosky = report_type == 'gosky'
        hostinfo = self._db.hostinfo
        version = req.args.get('version', None)
        version = TypeSafeLooseVersion(version) if version else None
        aliveThreshold = dt.datetime.now() - dt.timedelta(self._alive_threshold)
        unknown = req.args.get('unknown', '').lower() == 'yes'

        source = None
        if report_type == 'rulez':
            source = flask.request.args.get('source')
            if source:
                rulez, hosts = self._rulezSelector(source)
            deadline, deadlines = dt.timedelta.min, []
        else:
            deadline, deadlines = self._webDeadlines(prefix='last ')

        counts = [0, 0]
        collected = set()
        query = {'last_update': {'$gt': dt.datetime.now() - deadline}} if deadline != dt.timedelta.min else {}
        if hosts and len(hosts) < QUERY_HOSTS_LIMIT:
            # Construct dummy query in case of no query data provided yet.
            query['host'] = {'$in': list(hosts)}

        data = []
        started = time.time()
        if query:
            for doc in hostinfo.find(
                query,
                {'_id': False, 'host': True, 'last_update': True, 'gosky': True, 'skynet': True},
            ):
                gver = '.'.join(map(str, doc.get('gosky', {}).get('version', [])))
                skynet_info = doc.get('skynet', {})
                sver = str(skynet_info.get('version', None))
                ver = (gver if gver else None) if gosky else sver
                if ver == 'None':
                    continue
                if version and ver and version <= ver:
                    continue
                if not unknown and not hosts and not ver:
                    continue

                host = doc['host']
                if host.find('.') < 0:
                    host += '.yandex.ru'
                if hosts and host not in hosts:
                    continue

                alive = doc['last_update'] > aliveThreshold
                counts[0 if alive else 1] += 1
                if sum(counts) > RESULT_LIMIT:
                    break
                collected.add(host)
                data.append({
                    'host': host,
                    'gosky': gver,
                    'skynet': sver,
                    'alive': alive,
                    'lu': doc['last_update'],
                    'diff': dt.datetime.now() - doc['last_update'],
                })

            log.debug(
                'Database query returned %d hosts and finished in %.3gs.',
                sum(counts), time.time() - started
            )

        # List unknown hosts
        for host in hosts - collected if unknown else []:
            counts[1] += 1
            if sum(counts) > RESULT_LIMIT:
                break
            data.append({'host': host, 'gosky': None, 'skynet': None, 'alive': None, 'lu': None, 'diff': None})

        if gosky:
            aggregation = self._db.hostinfo.aggregate([{'$group': {'_id': '$gosky.version'}}])
            versions = list(map(
                lambda x: '.'.join(map(str, x['_id'])),
                filter(lambda x: x['_id'], aggregation)
            ))
        else:
            aggregation = self._db.hostinfo.aggregate([{'$group': {'_id': '$skynet.version'}}])
            versions = list(filter(None, map(lambda x: str(x['_id']), aggregation)))

        versions = list(map(lambda x: (x, True if x == 'None' else x == version), self._jinja_version_sort(versions)))
        return flask.render_template(
            'outdated_report.html',
            limit=RESULT_LIMIT,
            deadlines=deadlines,
            versions=versions,
            unknown=unknown,
            stats={
                'duration': deadline if deadline != dt.timedelta.min else None,
                'version': version,
                'total': sum(counts),
                'alive': counts[0],
                'dead': counts[1],
            },
            rulez=rulez,
            data=data,
            source=source
        )

    def webHBSState(self):
        class Avg(object):
            __slots__ = ['values']

            def __init__(self, initial=None):
                self.values = [initial] if initial is not None else []

            def __add__(self, other):
                if isinstance(other, Avg):
                    self.values += other.values
                else:
                    self.values.append(other)
                return self

            def __float__(self):
                return sum(self.values) / len(self.values)

        def fixavg(avg, json=False):
            for k, v in six.iteritems(avg):
                if isinstance(v, dict):
                    fixavg(v, json)
                elif isinstance(v, Avg):
                    avg[k] = float(v)
                elif json:
                    if isinstance(v, set):
                        avg[k] = list(v)
                    elif isinstance(v, dt.datetime):
                        avg[k] = str(v)
            return avg

        def recavg(doc, avg):
            if not isinstance(doc, dict):
                return

            for k, v in six.iteritems(doc):
                if isinstance(v, dict):
                    recavg(v, avg.setdefault(k, {}))
                elif isinstance(v, int):
                    avg[k] = avg.get(k, 0) + v
                elif isinstance(v, (float, Avg, )):
                    if k in ('busy', 'upsize', 'psize'):
                        prev = avg.get(k, Avg())
                        avg[k] = (prev if isinstance(prev, (float, Avg, )) else Avg(prev)) + v
                    elif k == 'uptime':
                        prev = avg.get(k, None)
                        avg[k] = min(prev, v) if prev is not None else v
                    else:
                        avg[k] = avg.get(k, 0.0) + v
                elif isinstance(v, (list, set, )):
                    avg[k] = avg.get(k, set()) | set(v)
                elif isinstance(v, dt.datetime):
                    avg['max_' + k] = max(avg.get('max_' + k, dt.datetime.min), v)
                    avg['min_' + k] = min(avg.get('min_' + k, dt.datetime.max), v)
                else:
                    avg[k] = v

        ret, avg = {}, {}
        for doc in self._db.hbs_state.find(
            {'host': re.compile(self.cfg.get('server_regexp', ''), re.IGNORECASE)},
            {'_id': False},
            sort=[('host', pymongo.ASCENDING)],
        ):
            recavg(doc, avg)
            plugs = doc['plugins']
            pavg = plugs.pop('!', {})
            for v in six.itervalues(plugs):
                recavg(v, pavg)
            plugs['!'] = pavg

            ret[doc['host']] = doc

        plugs = avg['plugins']
        pavg = plugs.pop('!', {})
        for v in six.itervalues(plugs):
            recavg(v, pavg)
        plugs['!'] = pavg
        ret['!'] = avg

        if flask.request.args.get('json') is not None or not flask.request.accept_mimetypes.accept_html:
            resp = flask.make_response(flask.json.dumps(fixavg(ret, True)))
            resp.headers['Content-Type'] = 'application/json'
            return resp

        return flask.render_template(
            'hbs.html',
            len=len,
            hosts=fixavg(ret),
            szfmt=util.size2str,
        )

    def webTimeSkew(self):
        hosts = {}
        DEADLINE = dt.timedelta(minutes=30)
        for doc in self._db.hostclockskewreport.find(
            {'last_update': {'$gt': dt.datetime.now() - DEADLINE}},
            {'_id': False},
        ):
            hosts[doc['host']] = doc

        hosts = list(map(
            lambda item: item[1],
            sorted(six.iteritems(hosts), key=lambda item: abs(item[1]['skew']), reverse=True),
        ))
        for h in hosts:
            h['skew'] = dt.timedelta(seconds=h['skew'])

        return flask.render_template(
            'time_skew.html',
            hosts=hosts,
        )

    def webPluginErrors(self):
        PREFIX = 'pluginerror_'
        deadline, deadlines = self._webDeadlines()
        collections = list(filter(lambda c: c.startswith(PREFIX), self._db.collection_names()))

        plug2err = {}
        query = {'last_update': {'$gt': dt.datetime.now() - deadline}} if deadline != dt.timedelta.min else {}
        for cname in collections:
            name = cname[len(PREFIX):]
            for doc in self._db[cname].find(query, {'_id': False}):
                errors = plug2err.setdefault(name, {})
                err = errors.setdefault(doc['sha1sum'], {})
                if not err:
                    err['hosts'] = [doc['host']]
                    for k in six.iterkeys(doc):
                        if k not in ['host', 'last_update', 'sha1sum']:
                            err[k] = doc[k]
                    tout = err.get('timeout', None)
                    if tout:
                        err['timeout'] = dt.timedelta(seconds=tout)
                else:
                    err['hosts'].append(doc['host'])

        for errs in six.itervalues(plug2err):
            for err in six.itervalues(errs):
                hosts = err['hosts']
                err['hosts'] = (len(hosts), sorted(hosts), )

        plugerr = sorted(
            [
                (
                    plug_name,
                    sorted(
                        filter(lambda item: item[1]['hosts'][0] > 1, six.iteritems(plug_errors)),
                        key=lambda item: item[1]['hosts'][0], reverse=True
                    ),
                )
                for plug_name, plug_errors in six.iteritems(plug2err)
            ],
            key=lambda item: sum(map(lambda item: item[1]['hosts'][0], item[1])),
            reverse=True
        )

        return flask.render_template(
            'plugin_errors.html',
            plugerr=plugerr,
            deadlines=deadlines,
        )

    @staticmethod
    def _sort(default):
        arg = flask.request.args.get('sort')
        if not arg:
            return default

        sort = []
        for s in arg.split(','):
            order = pymongo.ASCENDING
            if s[0] in ['+', '-']:
                order = pymongo.DESCENDING if s[0] == '-' else pymongo.ASCENDING
                s = s[1:]
            sort.append((s, order, ))
        return sort

    def getHostInfo(self, host):
        hostinfo = self._db.hostinfo
        hostversions = list(hostinfo.find({'host': {'$regex': '^' + host}}))
        return flask.render_template('hostinfo.html', host=host, hostversions=hostversions)

    def webVersions(self, version=None):
        req = flask.request
        hostinfo = self._db.hostinfo
        # FIXME: currently the default bgtask_name is hardcoded
        source = req.args.get('source', 'Skynet_rulez')
        rulez, hosts = (None, set()) if not req.args.get('rule') else self._rulezSelector(source)

        if req.args.get('json') is None and req.args.get('download') is None and req.accept_mimetypes.accept_html:
            # construct URL for downloading
            download_url = '/versions{}?{}{}{}{}download=1'.format(
                '/' + version if version else '',
                ('rule=' + req.args['rule'] + '&') if 'rule' in req.args else '',
                ('type=' + req.args['type'] + '&') if 'type' in req.args else '',
                ('alive=' + req.args['alive'] + '&') if 'alive' in req.args else
                ('died=' + req.args['died'] + '&') if 'died' in req.args else '',
                ('source=' + source + '&')
            )
            return flask.render_template(
                'versions.html',
                service=source.split('_')[0],
                rulez=rulez,
                keepLoading=True,
                download_url=download_url,
                source=source)

        query = {}
        aliveThreshold = dt.datetime.now() - dt.timedelta(seconds=self._alive_threshold)

        if req.args.get('died'):
            query['last_update'] = {'$lt': aliveThreshold}

        if req.args.get('alive'):
            query['last_update'] = {'$gt': aliveThreshold}

        if version:
            hostinfo.ensure_index('skynet.version')
            query['skynet.version'] = version

        if hosts:
            if len(hosts) < 10000:
                # Perform database-side hosts filtering in case of not so big amount of hosts listed.
                query['host'] = {'$in': list(hosts)}

        data = []
        now = dt.datetime.now()

        if version and version == 'no_skynet':
            # all hosts are dead, because there is not skynet yet
            if not req.args.get('alive') and self._bgp.skynetChecker:
                skynet_hosts_info = self._bgp.skynetChecker.info
                for host in skynet_hosts_info['clean_hosts']:
                    data.append([host] + ['no_skynet'] + [None] * 6 + [True])
        elif query or len(hosts) >= 10000:
            cur = hostinfo.find(
                query,
                'host gosky skynet last_update'.split(),
            )
            collected = set()

            for doc in cur:
                if hosts and doc['host'] not in hosts:
                    continue

                collected.add(doc['host'])

                info = doc.get('skynet', {})
                reason = ['', '']
                reason[1] = str(info.get('reason', ''))
                if reason[1]:
                    reason[1] += ' '

                new_services_info = info['skycore'].get('services', {}) if 'skycore' in info else {}
                old_services_info = info['srvmngr'].get('services', {}) if 'srvmngr' in info else {}

                if new_services_info or old_services_info:
                    failed_services = []
                    ok_services = []
                    for service, state in six.iteritems(new_services_info):
                        # dirty hack for invalid records in mongo (just ignore unexpected values)
                        if isinstance(state, dict) and 'state' in state:
                            if state['state'] in ['PRERUNNING', 'RUNNING', 'PREFAIL']:
                                ok_services.append((service, state['state']))
                            else:
                                failed_services.append((service, state['state']))

                    for service, state in six.iteritems(old_services_info):
                        if state:
                            ok_services.append((service, None))
                        else:
                            failed_services.append((service, None))

                    short_state = {
                        'STOPPED': 'SD',
                        'DEPENDENCIES_WAIT': 'DW',
                        'STARTING': 'ST',
                        'PRERUNNING': 'PR',
                        'RUNNING': 'RU',
                        'PREFAIL': 'PF',
                        'STOPPING': 'SG',
                        'KILLING': 'KI',
                        'CLEANUP': 'CU',
                        'COOLDOWN': 'CD'
                    }
                    for i, services in enumerate([ok_services, failed_services]):
                        reason[i] += '[{}]: '.format(len(services)) if services else ''
                        for service, state in sorted(services):
                            reason[i] += '[{}]{} '.format(short_state.get(state, 'UN'), service)

                try:
                    gosky = doc['gosky']['version']
                except KeyError:
                    gosky = 'Unknown'
                else:
                    if isinstance(gosky, list):
                        gosky = '{}.{}.{}'.format(*gosky)

                data.append([
                    self._shortify(doc['host']),
                    self._bgp[source].get_version(info),
                    gosky,
                    util.dt2str(info['installed']) if 'installed' in info else '',
                    util.dt2str(doc['last_update']),
                    util.td2str(now - doc['last_update']),
                    reason[0],
                    reason[1],
                    doc['last_update'] < aliveThreshold or reason[1]
                ])

            # List unknown hosts
            for host in hosts - collected:
                data.append([self._shortify(host)] + [None] * 7 + [True])

        self.log.debug('Versions list data set built in %.3gs', time.time() - time.mktime(now.timetuple()))
        if 'download' in req.args:
            resp = flask.make_response(formatHosts([d[0] for d in data],
                                                   useGrouping=False, yrFormat=False, addDomain=True, separator='\n'))
            filename = '{}{}{}{}hosts.csv'.format(
                version + '_' if version else 'all_',
                req.args['rule'].replace(' ', '_') + '_' if 'rule' in req.args else '',
                req.args['type'] + '_' if 'type' in req.args else '',
                'alive_' if 'alive' in req.args else
                'died_' if 'died' in req.args else '')
            resp.headers["Content-Disposition"] = "attachment; filename=%s" % filename
        else:
            resp = flask.make_response(flask.json.dumps(
                {'full_list': data, 'quick_list': formatHosts([d[0] for d in data], yrFormat=False, addDomain=True)}
            ))
            resp.headers['Content-Type'] = 'application/json'
        return resp

    def add_new_event(self, service, description, state):
        with self._eventsLock:
            self._events[service] = (description, state)

    def get_events(self):
        with self._eventsLock:
            events = self._events
            self._events = dict()
            return events
