package main

import (
	"bytes"
	"fmt"
	"io"
	"io/ioutil"
	"log"
	"os"
	"os/exec"
	"strings"
	"sync"
	"text/template"
	"time"

	"github.com/spf13/cobra"

	_ "a.yandex-team.ru/infra/hmserver/bootstrap/cockroach/scripts/templates"
	"a.yandex-team.ru/library/go/httputil/resource"
)

const (
	tmpRoot          = "/tmp/cockroach_bootstrap"
	cockroachVersion = "cockroach-v20.1.3.linux-amd64"
	templatesDir     = resource.Dir("/templates/")
)

func Execute() {
	var rootCmd = &cobra.Command{
		Use:   "cockroach-bootstrap",
		Short: "Bootstrap for cockroach cluster.",
	}
	l := log.New(os.Stdout, "", log.Lshortfile)
	l.SetPrefix("[" + time.Now().Format("2006-01-02 15:04:05") + "] ")

	systemdTemplateFilename := ""
	outDir := "cluster"
	redeployNode := ""
	certsDir := ""
	user := ""
	configPath := ""

	cmdBootstrap := &cobra.Command{
		Use:   "bootstrap",
		Short: "Bootstrap cluster",
		Args:  cobra.NoArgs,
		Run: func(cmd *cobra.Command, args []string) {
			config, err := ReadConfig(configPath)
			if err != nil {
				er(err, l)
			}
			err = createClusterCerts(config.Nodes, []string{"root"}, outDir, l)
			if err != nil {
				er(err, l)
			}
			for _, node := range config.Nodes {
				err = deployNode(user, config, outDir+"/"+node, node, systemdTemplateFilename, l)
				if err != nil {
					er(err, l)
				}
			}
			firstNode := config.Nodes[0]
			err = initCockroach(l, firstNode, outDir+"/"+firstNode)
			if err != nil {
				er(err, l)
			}
		},
	}

	cmdRedeployNode := &cobra.Command{
		Use:   "redeploy-node",
		Short: "Redeploy cockroach node",
		Args:  cobra.NoArgs,
		Run: func(cmd *cobra.Command, args []string) {
			config, err := ReadConfig(configPath)
			if err != nil {
				er(err, l)
			}
			err = stopNode(user, redeployNode, l)
			if err != nil {
				er(err, l)
			}
			err = deployNode(user, config, certsDir, redeployNode, systemdTemplateFilename, l)
			if err != nil {
				er(err, l)
			}
		},
	}

	cmdRedeployNode.PersistentFlags().StringVar(&redeployNode, "node", "kek1-0000.search.yandex.net", "Node to redeploy")
	cmdRedeployNode.PersistentFlags().StringVar(&certsDir, "certs", "certs", "Cockroach node certs")
	cmdBootstrap.PersistentFlags().StringVar(&outDir, "out", "cluster", "Output credentials directory")
	cmdBootstrap.PersistentFlags().StringVar(&systemdTemplateFilename, "systemd-tmpl", "templates/cockroachdb.service.tmpl", "Systemd template for cockroachdb node")
	rootCmd.PersistentFlags().StringVar(&user, "user", "root", "SSH user")
	rootCmd.PersistentFlags().StringVar(&configPath, "path", "", "Config path")
	rootCmd.AddCommand(cmdBootstrap)
	rootCmd.AddCommand(cmdRedeployNode)
	if err := rootCmd.Execute(); err != nil {
		l.Println(err)
		os.Exit(1)
	}
}

func deployNode(user string, config *ClusterConfig, certsDir, node, systemdTemplateFilename string, l *log.Logger) error {
	systemdService, err := createSystemdTemplate(config, node, systemdTemplateFilename)
	if err != nil {
		return err
	}
	err = deploySystemdService(user, l, node, systemdService)
	if err != nil {
		return err
	}
	err = deployCockroach(user, l, node, certsDir, config.DataDir)
	if err != nil {
		return err
	}
	return err
}

func stopNode(user, node string, l *log.Logger) error {
	_ = execCmd(l, "ssh", user+"@"+node, "sudo systemctl stop cockroachdb")
	return nil
}

func reloadSystemd(user, node string, l *log.Logger) error {
	_ = execCmd(l, "ssh", user+"@"+node, "sudo systemctl daemon-reload")
	return nil
}

func deploySystemdService(user string, l *log.Logger, node string, service string) error {
	data := struct {
		ServiceTemplate string
	}{
		ServiceTemplate: service,
	}
	err := execScriptSSH(l, user, node, "deploy-systemd-service.sh.tmpl", data)
	return err
}

func deployCockroach(user string, l *log.Logger, node, certsDir, dataDir string) error {
	err := execCmd(l, "scp", "-r", certsDir, fmt.Sprintf("%s@%s:", user, node))
	if err != nil {
		return err
	}
	data := struct {
		CockroachVersion string
		NodeName         string
		DataDir          string
	}{
		CockroachVersion: cockroachVersion,
		NodeName:         node,
		DataDir:          dataDir,
	}
	_ = execScriptSSH(l, user, node, "deploy-cockroach.sh.tmpl", data)
	_ = reloadSystemd(user, node, l)
	err = execCmd(l, "ssh", user+"@"+node, "sudo systemctl start cockroachdb")
	if err != nil {
		return err
	}
	err = execCmd(l, "ssh", user+"@"+node, "sudo systemctl enable cockroachdb")
	if err != nil {
		return err
	}
	return nil
}

func initCockroach(l *log.Logger, node string, certsDir string) error {
	err := execCmd(l, "cockroach", "init", fmt.Sprintf("--certs-dir=%s", certsDir), fmt.Sprintf("--host=%s", node))
	if err != nil {
		return err
	}
	return nil
}

func createSystemdTemplate(config *ClusterConfig, currentNode, templateFile string) (string, error) {
	type TmplData struct {
		Nodes        string
		CurrentNode  string
		MaxSQLMemory string
		DataDir      string
	}
	file, err := os.Open(templateFile)
	if err != nil {
		return "", err
	}
	defer file.Close()
	tmpl, err := ioutil.ReadAll(file)
	if err != nil {
		return "", err
	}
	t := template.Must(template.New("data").Parse(string(tmpl)))
	buf := new(bytes.Buffer)
	err = t.Execute(buf, TmplData{strings.Join(config.Nodes, ","), currentNode, config.MaxSQLMemory, config.DataDir})
	return buf.String(), err
}

func execCmd(l *log.Logger, name string, arg ...string) error {
	l.Println(name + " " + strings.Join(arg, " "))
	cmd := exec.Command(name, arg...)
	out, err := cmd.CombinedOutput()
	l.Println(string(out))
	if err != nil {
		return err
	}
	return nil
}

func execScript(l *log.Logger, script *bytes.Buffer) error {
	l.Println("executing bash script: \n" + script.String())
	bash := exec.Command("bash")
	stdin, err := bash.StdinPipe()
	if err != nil {
		return err
	}
	stdout, err := bash.StdoutPipe()
	if err != nil {
		return err
	}
	stderr, err := bash.StderrPipe()
	if err != nil {
		return err
	}
	wait := sync.WaitGroup{}
	wait.Add(3)
	go func() {
		_, _ = io.Copy(stdin, script)
		_ = stdin.Close()
		wait.Done()
	}()
	go func() {
		_, _ = io.Copy(os.Stdout, stdout)
		wait.Done()
	}()
	go func() {
		_, _ = io.Copy(os.Stderr, stderr)
		wait.Done()
	}()
	err = bash.Start()
	if err != nil {
		return err
	}
	wait.Wait()
	err = bash.Wait()
	if err != nil {
		return err
	}
	return nil
}

func execScriptSSH(l *log.Logger, user, node, scriptTemplate string, data interface{}) error {
	tmpl, err := getTemplate(scriptTemplate)
	if err != nil {
		return err
	}
	buf := &bytes.Buffer{}
	t := template.Must(template.New("data").Parse(tmpl))
	err = t.Execute(buf, data)
	if err != nil {
		return err
	}
	err = execCmd(l, "ssh", user+"@"+node, fmt.Sprintf("sudo /bin/sh `%s`", buf.String()))
	if err != nil {
		return err
	}
	return nil
}

func er(err error, l *log.Logger) {
	l.Println(err)
	os.Exit(1)
}

func getTemplate(filename string) (string, error) {
	tmplFile, err := templatesDir.Open(filename)
	if err != nil {
		return "", err
	}
	tmpl, err := ioutil.ReadAll(tmplFile)
	if err != nil {
		return "", err
	}
	return string(tmpl), nil
}
