package main

import (
	"bytes"
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"os/exec"
	"strings"
	"text/template"
	"time"

	"github.com/spf13/cobra"

	_ "a.yandex-team.ru/infra/hmserver/bootstrap/hm-reporter/scripts/templates"
	"a.yandex-team.ru/library/go/httputil/resource"
)

const templatesDir = resource.Dir("/templates/")

func Execute() {
	var rootCmd = &cobra.Command{
		Use:   "hm-reporter-bootstrap",
		Short: "Bootstrap for cockroach cluster.",
	}
	l := log.New(os.Stdout, "", log.Lshortfile)
	l.SetPrefix("[" + time.Now().Format("2006-01-02 15:04:05") + "] ")

	systemdServiceFilename := ""
	envFile := ""
	binaryPath := ""
	user := ""

	cmdStart := &cobra.Command{
		Use:   "bootstrap <node 1> <node 2> ... <node n>",
		Short: "Bootstrap cluster",
		Run: func(cmd *cobra.Command, args []string) {
			for _, node := range args {
				err := deployNode(node, user, envFile, systemdServiceFilename, binaryPath, l)
				if err != nil {
					er(err, l)
				}
			}
		},
	}

	cmdStart.PersistentFlags().StringVar(&systemdServiceFilename, "systemd-tmpl", "templates/hm-reporter-server.service", "Systemd file for hm-reporter-server")
	cmdStart.PersistentFlags().StringVar(&envFile, "env-file", "templates/hm-reporter-server.conf", "hm-reporter-server env file")
	cmdStart.PersistentFlags().StringVar(&binaryPath, "binary-path", "hm-reporter-server", "Path to hm-reporter-server")
	cmdStart.PersistentFlags().StringVar(&user, "user", "root", "SSH user")
	rootCmd.AddCommand(cmdStart)
	if err := rootCmd.Execute(); err != nil {
		l.Println(err)
		os.Exit(1)
	}
}

func deployNode(node, user, envFile, systemdServiceFilename, binaryPath string, l *log.Logger) error {
	err := execScriptSSH(l, user, node, "prepare-node.sh.tmpl", struct{}{})
	if err != nil {
		return err
	}
	err = deployService(l, systemdServiceFilename, envFile, node, binaryPath)
	if err != nil {
		return err
	}
	err = execScriptSSH(l, user, node, "start-node.sh.tmpl", struct{}{})
	if err != nil {
		return err
	}
	return err
}

func deployService(l *log.Logger, systemdFile, envFile, node, binaryPath string) error {
	err := execCmd(l, "scp", systemdFile, fmt.Sprintf("root@%s:/etc/systemd/system/hm-reporter-server.service", node))
	if err != nil {
		return err
	}
	err = execCmd(l, "scp", envFile, fmt.Sprintf("root@%s:/etc/hm-reporter-server.conf", node))
	if err != nil {
		return err
	}
	err = execCmd(l, "scp", binaryPath, fmt.Sprintf("root@%s:/opt/hm-reporter-server/", node))
	if err != nil {
		return err
	}
	return err
}

func execCmd(l *log.Logger, name string, arg ...string) error {
	l.Println(name + " " + strings.Join(arg, " "))
	cmd := exec.Command(name, arg...)
	out, err := cmd.CombinedOutput()
	l.Println(string(out))
	if err != nil {
		return err
	}
	return nil
}

func er(err error, l *log.Logger) {
	l.Println(err)
	os.Exit(1)
}

func execScriptSSH(l *log.Logger, user, node, scriptTemplate string, data interface{}) error {
	fmt.Println(scriptTemplate)
	tmpl, err := getTemplate(scriptTemplate)
	fmt.Println(tmpl)
	if err != nil {
		return err
	}
	buf := &bytes.Buffer{}
	t := template.Must(template.New("data").Parse(tmpl))
	err = t.Execute(buf, data)
	if err != nil {
		return err
	}
	err = execCmd(l, "ssh", user+"@"+node, fmt.Sprintf("sudo /bin/sh `%s`", buf.String()))
	if err != nil {
		return err
	}
	return nil
}

func getTemplate(filename string) (string, error) {
	tmplFile, err := templatesDir.Open(filename)
	if err != nil {
		return "", err
	}
	tmpl, err := ioutil.ReadAll(tmplFile)
	if err != nil {
		return "", err
	}
	return string(tmpl), nil
}
