/* Multi repository sync mode */
package main

import (
	"errors"
	"fmt"
	"github.com/spf13/cobra"
	"log"
	"os"

	"a.yandex-team.ru/infra/hmserver/pkg/app"
	"a.yandex-team.ru/infra/hmserver/pkg/bitbucket"
	"a.yandex-team.ru/infra/hmserver/pkg/sync"
)

type multiConfig struct {
	project  string
	role     string
	repos    []string
	token    string
	listener string
}

func validateMultiConfig(mc *multiConfig) error {
	if mc.token == "" {
		return errors.New("no token specified in BB_OAUTH_TOKEN")
	}
	return nil
}

func fixUpBranch(branch string) string {
	switch branch {
	case "prestable":
		return "develop"
	case "sas":
		return "master_sas"
	case "man":
		return "master_man"
	case "vla":
		return "master_vla"
	case "msk":
		return "master_msk"
	default:
		return branch
	}
}

func runMulti(mc *multiConfig) error {
	// We log through systemd-journal, so no need to format time ourselves
	l := log.New(os.Stdout, "", log.Lshortfile)
	if err := validateMultiConfig(mc); err != nil {
		l.Println("Error:", err)
		os.Exit(1)
	}
	remotes := make(map[string]bitbucket.Bitbucket)
	for _, name := range mc.repos {
		branch := mc.role
		// Fix up for original monolithic repo compatibility
		if name == "saltstack" {
			branch = fixUpBranch(branch)
		}
		if _, ok := remotes[name]; ok {
			return fmt.Errorf("duplicate remote '%s'", name)
		}
		l.Printf("Using %s:%s as remote...", name, branch)
		remotes[name] = bitbucket.NewProduction(mc.project, name, branch, mc.token)
	}
	bbs := sync.NewMultiBB(l, remotes)
	return app.NewProduction(l, mc.listener, bbs).Run()
}

func registerMulti(root *cobra.Command) {
	mc := &multiConfig{}
	cmd := &cobra.Command{
		Use:   "multi",
		Short: "Run in production mode with multiple repositories sync",
		Long: `Serves hostman repositories from bitbucket.

**Requires BB_OAUTH_TOKEN in environment**.

Accepts multiple repositories, thus whole argument list can look like this:

hmserver --role sas --repos saltstack,sysdev,hostman.

Merge rules:
  * repository named "saltstack" files are merged as is
  branches are fixed up, develop -> prestable, master_sas -> sas, etc.
  * all other repository files are added to resulting data prefixed with remote name 
  e.g sysdev/porto-daemons.d`,
		RunE: func(cmd *cobra.Command, args []string) error {
			// Get token from env
			mc.token = os.Getenv("BB_OAUTH_TOKEN")
			return runMulti(mc)
		},
	}
	flags := cmd.Flags()
	flags.StringVar(&mc.listener, "address", "[::]:8080", "address to listen to")
	flags.StringVar(&mc.project, "project", "RTCSALT", "bitbucket project")
	flags.StringVar(&mc.role, "role", "", "role (used as branch name), e.g. 'sas' or 'prestable'")
	flags.StringSliceVar(&mc.repos, "remotes", nil, "remote repo names, can be ',' separated")
	root.AddCommand(cmd)
}
