package main

import (
	"errors"
	"log"
	"os"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/infra/hmserver/pkg/app"
	"a.yandex-team.ru/infra/hmserver/pkg/bitbucket"
	"a.yandex-team.ru/infra/hmserver/pkg/sync"
)

type productionConfig struct {
	project  string
	repo     string
	branch   string
	token    string
	listener string
}

var c productionConfig

func validateProductionConfig(c *productionConfig) error {
	if c.token == "" {
		return errors.New("no token specified in BB_OAUTH_TOKEN")
	}
	if c.branch == "" {
		return errors.New("no branch specified")
	}
	return nil
}

func runProduction(c *productionConfig) error {
	// We log through systemd-journal, so no need to format time ourselves
	l := log.New(os.Stdout, "", log.Lshortfile)
	if err := validateProductionConfig(c); err != nil {
		l.Println("Error:", err)
		os.Exit(1)
	}
	bb := bitbucket.NewProduction(
		c.project, c.repo, c.branch,
		c.token)
	bbs := sync.NewBB(l, bb)
	return app.NewProduction(l, c.listener, bbs).Run()
}

func registerProduction(root *cobra.Command) {
	cmd := &cobra.Command{
		Use:   "production",
		Short: "Run in production mode",
		Long:  `Serves salt repo from bitbucket. Requires BB_OAUTH_TOKEN in environment`,
		RunE: func(cmd *cobra.Command, args []string) error {
			// Get token from env
			c.token = os.Getenv("BB_OAUTH_TOKEN")
			return runProduction(&c)
		},
	}
	cmd.Flags().StringVar(&c.listener, "address", "[::]:8080", "address to listen to")
	cmd.Flags().StringVar(&c.project, "project", "RTCSALT", "bitbucket project")
	cmd.Flags().StringVar(&c.repo, "repo", "saltstack", "repo name")
	cmd.Flags().StringVar(&c.branch, "branch", "", "branch name, e.g. master_sas")
	root.AddCommand(cmd)
}
