package api

import (
	"net/http"
	"strconv"
	"time"

	"a.yandex-team.ru/infra/hmserver/pkg/sync"
	"a.yandex-team.ru/library/go/yandex/unistat"
	"a.yandex-team.ru/library/go/yandex/unistat/aggr"
)

type SaltRepo struct {
	store            sync.Store
	okCount          *unistat.Numeric
	failCount        *unistat.Numeric
	modifiedCount    *unistat.Numeric
	notModifiedCount *unistat.Numeric
}

func NewSaltRepo(store sync.Store) *SaltRepo {
	ok := unistat.NewNumeric(
		"salt-repo-zip-ok",
		10, aggr.Counter(), unistat.Sum)
	fail := unistat.NewNumeric(
		"salt-repo-zip-fail",
		10, aggr.Counter(), unistat.Sum)
	mod := unistat.NewNumeric(
		"salt-repo-zip-modified",
		10, aggr.Counter(), unistat.Sum)
	notModified := unistat.NewNumeric(
		"salt-repo-zip-not-modified",
		10, aggr.Counter(), unistat.Sum)
	unistat.Register(ok)
	unistat.Register(fail)
	unistat.Register(mod)
	unistat.Register(notModified)

	return &SaltRepo{
		store:            store,
		okCount:          ok,
		failCount:        fail,
		modifiedCount:    mod,
		notModifiedCount: notModified,
	}
}

// condResult is the result of an HTTP request precondition check.
// See https://tools.ietf.org/html/rfc7232 section 3.
type condResult int

const (
	condNone condResult = iota
	condTrue
	condFalse
)

// isZeroTime reports whether t is obviously unspecified (either zero or Unix()=0).
func isZeroTime(t time.Time) bool {
	return t.IsZero() || t.Equal(time.Unix(0, 0))
}

func checkIfModifiedSince(r *http.Request, modtime time.Time) condResult {
	if r.Method != "GET" && r.Method != "HEAD" {
		return condNone
	}
	ims := r.Header.Get("If-Modified-Since")
	if ims == "" || isZeroTime(modtime) {
		return condNone
	}
	t, err := http.ParseTime(ims)
	if err != nil {
		return condNone
	}
	// The Date-Modified header truncates sub-second precision, so
	// use mtime < t+1s instead of mtime <= t to check for unmodified.
	if modtime.Before(t.Add(1 * time.Second)) {
		return condFalse
	}
	return condTrue
}

func writeNotModified(w http.ResponseWriter) {
	// RFC 7232 section 4.1:
	// a sender SHOULD NOT generate representation metadata other than the
	// above listed fields unless said metadata exists for the purpose of
	// guiding cache updates (e.g., Last-Modified might be useful if the
	// response does not have an ETag field).
	h := w.Header()
	delete(h, "Content-Type")
	delete(h, "Content-Length")
	if h.Get("Etag") != "" {
		delete(h, "Last-Modified")
	}
	w.WriteHeader(http.StatusNotModified)
}

func (s *SaltRepo) GetZip(w http.ResponseWriter, r *http.Request) {
	z, err := s.store.GetZip()
	if err != nil {
		s.failCount.Update(1)
		w.WriteHeader(500)
		_, _ = w.Write([]byte(err.Error()))
		return
	}
	if checkIfModifiedSince(r, z.Timestamp) == condFalse {
		s.notModifiedCount.Update(1)
		s.okCount.Update(1)
		writeNotModified(w)
		return
	}
	s.modifiedCount.Update(1)
	modtime := z.Timestamp.UTC().Format(http.TimeFormat)
	w.Header().Set("Last-Modified", modtime)
	w.Header().Set("Content-Type", "application/zip")
	w.Header().Set("Content-Length", strconv.Itoa(len(z.Content)))
	w.Header().Set("Content-Disposition", `attachment; filename="salt-repo.zip"`)
	w.Header().Set("X-Commit-Id", z.CommitID)
	_, _ = w.Write(z.Content)
	s.okCount.Update(1)
}

func (s *SaltRepo) Register(bind func(method, pattern string, h http.HandlerFunc)) {
	bind("GET", "/v1/salt/zip", s.GetZip)
}
