package app

import (
	"context"
	"log"
	"net/http"
	"os"
	"os/signal"
	"syscall"
	"time"

	"a.yandex-team.ru/infra/hmserver/pkg/api"
	"a.yandex-team.ru/infra/hmserver/pkg/httpserver"
	"a.yandex-team.ru/infra/hmserver/pkg/sync"
	"a.yandex-team.ru/infra/hmserver/pkg/tick"
)

type Application struct {
	l  *log.Logger
	hs *httpserver.Server
	ss sync.Sync
}

func NewProduction(l *log.Logger, addr string, ss sync.Sync) *Application {
	hs := httpserver.NewHTTP(addr)
	repoAPI := api.NewSaltRepo(ss)
	repoAPI.Register(hs.Bind)
	unistatAPI := api.NewUnistat(l)
	unistatAPI.Register(hs.Bind)
	app := &Application{
		l:  l,
		hs: hs,
		ss: ss,
	}
	return app
}

func (a *Application) Run() error {
	a.l.Println("Starting hm-server...")
	ctx, cancel := context.WithCancel(context.Background())
	sigs := make(chan os.Signal, 1)
	signal.Notify(sigs, syscall.SIGINT, syscall.SIGTERM)
	go func() {
		<-sigs
		cancel()
	}()
	// Fail on start if sync failed
	a.l.Println("Running sync...")
	if err := a.ss.Sync(ctx); err != nil {
		a.l.Println("Sync failed:", err)
		return err
	}
	p := tick.NewPeriodic(a.l, "repo", 1*time.Minute, a.ss)
	go func() {
		_ = p.Run(ctx)
	}()
	a.l.Printf("Starting HTTP server on %s...", a.hs.Addr())
	if err := a.hs.Run(ctx); err != http.ErrServerClosed {
		a.l.Println("HTTP server failed: ", err)
		return err
	}
	a.l.Println("Exiting...")
	return nil
}
