package bitbucket

import (
	"bytes"
	"context"
	"encoding/base64"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"time"
)

const (
	ProductionURL = "https://bb.yandex-team.ru"
	OauthUser     = "x-oauth-token"
	userAgent     = "infra.ya_salt.httpserver"
)

type Client struct {
	project    string
	repo       string
	branch     string
	baseURL    string
	basicAuth  string
	httpClient *http.Client
}

func NewProduction(project, repo, branch, token string) *Client {
	// Authentication via token is performed with basicAuth hack.
	h := []byte(OauthUser + ":" + token)
	return &Client{
		project:   project,
		repo:      repo,
		branch:    branch,
		baseURL:   ProductionURL,
		basicAuth: "Basic " + base64.StdEncoding.EncodeToString(h),
		httpClient: &http.Client{
			Timeout: 30 * time.Second,
		},
	}
}

func (bb *Client) newGET(url string) (*http.Request, error) {
	u := bb.baseURL + url
	req, err := http.NewRequest("GET", u, nil)
	if err != nil {
		return nil, err
	}
	req.Header.Set("Authorization", bb.basicAuth)
	req.Header.Set("User-Agent", userAgent)
	return req, nil
}

func (bb *Client) formatLastCommitURL() string {
	rel := &url.URL{
		Path: "/rest/api/latest/projects/" + bb.project + "/repos/" + bb.repo + "/commits",
	}
	q := rel.Query()
	q.Set("limit", "1")
	q.Set("until", "refs/heads/"+bb.branch)
	rel.RawQuery = q.Encode()
	return rel.String()
}

func (bb *Client) GetLastCommit(ctx context.Context) (*CommitInfo, error) {
	req, err := bb.newGET(bb.formatLastCommitURL())
	if err != nil {
		return nil, err
	}
	req = req.WithContext(ctx)
	req.Header.Set("Accept", "application/json")
	resp, err := bb.httpClient.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if resp.StatusCode != 200 {
		return nil, fmt.Errorf("http request failed with %s", resp.Status)
	}
	l := struct {
		Values []*CommitInfo `json:"values"`
	}{}
	if err := json.NewDecoder(resp.Body).Decode(&l); err != nil {
		return nil, fmt.Errorf("failed to decode response: %s", err)
	}
	if len(l.Values) == 0 {
		return nil, fmt.Errorf("no commits in requested branch")
	}
	return l.Values[0], nil
}

func (bb *Client) formatZipURL(at string) string {
	rel := &url.URL{
		Path: "/rest/api/latest/projects/" + bb.project + "/repos/" + bb.repo + "/archive",
	}
	q := rel.Query()
	q.Set("at", at)
	q.Set("format", "zip")
	rel.RawQuery = q.Encode()
	return rel.String()
}

func (bb *Client) ZipArchive(ctx context.Context, at string) ([]byte, error) {
	req, err := bb.newGET(bb.formatZipURL(at))
	if err != nil {
		return nil, err
	}
	req = req.WithContext(ctx)
	resp, err := bb.httpClient.Do(req)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()
	if resp.StatusCode != 200 {
		return nil, fmt.Errorf("http request failed with %s", resp.Status)
	}
	buf := bytes.Buffer{}
	if _, err := buf.ReadFrom(resp.Body); err != nil {
		return nil, fmt.Errorf("failed to read response: %s", err)
	}
	return buf.Bytes(), err
}
