package api

import (
	pb "a.yandex-team.ru/infra/hmserver/proto"
	"bytes"
	"crypto/tls"
	"encoding/json"
	"fmt"
	"github.com/golang/protobuf/proto"
	"golang.org/x/net/publicsuffix"
	"io/ioutil"
	"net/http"
	"net/http/cookiejar"
	"time"
)

type API struct {
	addr string
	c    *httpClient
}

func NewAPI(addr string) (*API, error) {
	c, err := newHTTPClient()
	if err != nil {
		return nil, err
	}
	return &API{addr, c}, err
}

func (a *API) GetNodes() (*NodesResponse, error) {
	url := fmt.Sprintf("%s/_status/nodes", a.addr)
	resp, err := a.c.GetRequest(url, make(Params), Headers{"Accept": "application/json"})
	if err != nil {
		return nil, err
	}
	jsonResp := &NodesResponse{}
	err = json.Unmarshal(resp, jsonResp)
	return jsonResp, err
}
func (a *API) GetRanges() (*RangesResponse, error) {
	url := fmt.Sprintf("%s/_status/ranges/", a.addr)
	resp, err := a.c.GetRequest(url, make(Params), Headers{"Accept": "application/json"})
	if err != nil {
		return nil, err
	}
	jsonResp := &RangesResponse{}
	err = json.Unmarshal(resp, jsonResp)
	return jsonResp, err
}

func (a *API) Login(username, password, addr string) error {
	req := &pb.UserLoginRequest{
		Username: username,
		Password: password,
	}
	marshaledReq, err := proto.Marshal(req)
	if err != nil {
		return err
	}
	_, err = a.c.PostRequest(fmt.Sprintf("%s/login", addr), marshaledReq, make(Params), Headers{"Content-Type": "application/x-protobuf"})
	if err != nil {
		return err
	}
	return nil
}

type httpClient struct {
	c *http.Client
}

func newHTTPClient() (*httpClient, error) {
	jar, err := cookiejar.New(&cookiejar.Options{PublicSuffixList: publicsuffix.List})
	if err != nil {
		return nil, err
	}
	tr := &http.Transport{
		MaxIdleConns:          3,
		IdleConnTimeout:       10 * time.Minute,
		TLSHandshakeTimeout:   10 * time.Second,
		ExpectContinueTimeout: 1 * time.Second,
		TLSClientConfig:       &tls.Config{InsecureSkipVerify: true},
	}
	c := &http.Client{
		Timeout:   5 * time.Second,
		Transport: tr,
		Jar:       jar,
	}
	return &httpClient{c}, nil
}

func (c *httpClient) request(httpMethod string, url string, data []byte, params Params, headers Headers) (resp []byte, err error) {
	httpReq, err := http.NewRequest(httpMethod, url, bytes.NewBuffer(data))
	if err != nil {
		return nil, err
	}
	values := httpReq.URL.Query()
	for k, v := range params {
		if v == "" {
			continue
		}
		values.Add(k, v)
	}
	httpReq.URL.RawQuery = values.Encode()
	for k, v := range headers {
		httpReq.Header.Add(k, v)
	}
	httpResp, err := c.c.Do(httpReq)
	if err != nil {
		return nil, err
	}
	defer func() {
		_ = httpResp.Body.Close()
	}()
	resp, err = ioutil.ReadAll(httpResp.Body)
	if err != nil {
		return nil, fmt.Errorf("failed to read response: %s", err.Error())
	}
	if httpResp.StatusCode != 200 {
		return nil, fmt.Errorf("bad response code: %d, message: %s", httpResp.StatusCode, string(resp))
	}
	return resp, nil
}

type Params map[string]string
type Headers map[string]string

func (c *httpClient) GetRequest(uri string, params Params, headers Headers) (resp []byte, err error) {
	return c.request(http.MethodGet, uri, make([]byte, 0), params, headers)
}

func (c *httpClient) PutRequest(uri string, data []byte, params Params, headers Headers) (resp []byte, err error) {
	return c.request(http.MethodPut, uri, data, params, headers)
}

func (c *httpClient) PostRequest(uri string, data []byte, params Params, headers Headers) (resp []byte, err error) {
	return c.request(http.MethodPost, uri, data, params, headers)
}

func (c *httpClient) DeleteRequest(uri string, data []byte, params Params, headers Headers) (resp []byte, err error) {
	return c.request(http.MethodDelete, uri, data, params, headers)
}

func (c *httpClient) PatchRequest(uri string, data []byte, params Params, headers Headers) (resp []byte, err error) {
	return c.request(http.MethodPatch, uri, data, params, headers)
}
