package monitoring

import (
	"a.yandex-team.ru/infra/hmserver/pkg/cockroach/api"
	"a.yandex-team.ru/infra/hmserver/pkg/metrics"
	"a.yandex-team.ru/infra/hmserver/pkg/yasmclient"
	"context"
	"log"
)

type CockroachMetrics struct {
	liveNodesCount        *metrics.Gauge
	totalNodesCount       *metrics.Gauge
	collectRequestFails   *metrics.Gauge
	unavailableRanges     *metrics.Gauge
	underreplicatedRanges *metrics.Gauge
	totalRanges           *metrics.Gauge
}

const (
	CockroachNodeStatusOK             = 3
	CockroachNodeStatusDecommissioned = 5
)

func CollectMetrics(c *api.API, username, password, addr string, mtrcs *CockroachMetrics) error {
	err := c.Login(username, password, addr)
	if err != nil {
		return err
	}
	nodes, err := c.GetNodes()
	if err != nil {
		return err
	}
	ranges, err := c.GetRanges()
	if err != nil {
		return err
	}
	totalNodesCount := 0
	liveNodesCount := 0
	unavailableRanges := 0
	underreplicatedRanges := 0
	totalRanges := 0
	for _, status := range nodes.LivenessByNodeID {
		if status != CockroachNodeStatusDecommissioned {
			totalNodesCount += 1
		}
		if status == CockroachNodeStatusOK {
			liveNodesCount += 1
		}
	}
	for _, r := range ranges.Ranges {
		if r.Problems.Unavailable {
			unavailableRanges++
		}
		if r.Problems.Underreplicated {
			underreplicatedRanges++
		}
		totalRanges++
	}
	mtrcs.liveNodesCount.Set(float64(liveNodesCount))
	mtrcs.totalNodesCount.Set(float64(totalNodesCount))
	mtrcs.unavailableRanges.Set(float64(unavailableRanges))
	mtrcs.underreplicatedRanges.Set(float64(underreplicatedRanges))
	mtrcs.totalRanges.Set(float64(totalRanges))
	return nil
}

func CollectCockroachSignals(ctx context.Context, l *log.Logger, c *api.API, username, password, addr string) []yasmclient.YasmValue {
	mtrcs := &CockroachMetrics{
		liveNodesCount:        metrics.NewGauge("hm-cockroach-live-nodes-count_tnnn"),
		totalNodesCount:       metrics.NewGauge("hm-cockroach-total-nodes-count_tnnn"),
		unavailableRanges:     metrics.NewGauge("hm-cockroach-unavailable-ranges_txxx"),
		underreplicatedRanges: metrics.NewGauge("hm-cockroach-underreplicated-ranges_txxx"),
		totalRanges:           metrics.NewGauge("hm-cockroach-total-ranges_tnnn"),
		collectRequestFails:   metrics.NewGauge("hm-cockroach-collect-request-fails_txxx"),
	}
	l.Println("Collecting metrics from cockroach...")
	err := CollectMetrics(c, username, password, addr, mtrcs)
	if err != nil {
		mtrcs.collectRequestFails.Set(1)
		l.Printf("Failed to collect metrics from cockroach: %s\n", err)
	} else {
		l.Println("Successfully collected cockroach metrics")
	}
	return []yasmclient.YasmValue{
		mtrcs.collectRequestFails.FmtPush(),
		mtrcs.totalNodesCount.FmtPush(),
		mtrcs.liveNodesCount.FmtPush(),
		mtrcs.unavailableRanges.FmtPush(),
		mtrcs.underreplicatedRanges.FmtPush(),
		mtrcs.totalRanges.FmtPush(),
	}
}
