package client

import (
	yasaltpb "a.yandex-team.ru/infra/hostctl/proto"
	"context"
	"google.golang.org/protobuf/types/known/timestamppb"
	"io"
	"math/rand"
	"time"

	"google.golang.org/grpc"

	pb "a.yandex-team.ru/infra/hmserver/proto"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
)

type Reporter struct {
	getAddr func() string
}

func NewClusterReporter(addresses []string) *Reporter {
	r := rand.New(rand.NewSource(time.Now().Unix()))
	return &Reporter{
		getAddr: func() string {
			return addresses[r.Intn(len(addresses))]
		},
	}
}

func NewReporter(addr string) *Reporter {
	return &Reporter{
		getAddr: func() string {
			return addr
		},
	}
}

type requestEnv struct {
	conn   *grpc.ClientConn
	ctx    context.Context
	client pb.ReporterClient
	cancel context.CancelFunc
	l      log.Logger
}

func createEnv(addr string) requestEnv {
	z, _ := zap.New(zap.ConsoleConfig(log.InfoLevel))
	conn, err := grpc.Dial(addr, grpc.WithInsecure())
	if err != nil {
		z.Fatalf("did not connect: %v", err)
	}
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*5)
	c := pb.NewReporterClient(conn)
	return requestEnv{
		conn:   conn,
		ctx:    ctx,
		client: c,
		cancel: cancel,
		l:      z,
	}
}

func (rc *Reporter) SendReport(rs []*pb.Unit, include *yasaltpb.HostInfo, lastStatusChange *timestamppb.Timestamp) error {
	e := createEnv(rc.getAddr())
	defer e.conn.Close()
	defer e.cancel()
	_, err := e.client.HandleReport(e.ctx, &pb.HandleReportRequest{Units: rs, HostInfo: include, LastStatusChange: lastStatusChange})
	return err
}

func (rc *Reporter) GetReports(u string, n string, s string) ([]*pb.Unit, error) {
	e := createEnv(rc.getAddr())
	defer e.conn.Close()
	defer e.cancel()
	resp, err := e.client.GetReports(e.ctx, &pb.GetReportsRequest{Node: n, Unit: u, Stage: s, Limit: 100, Offset: 0})
	if err != nil {
		return nil, err
	}
	return resp.Reports, nil
}

func (rc *Reporter) GetHost(h string) (*yasaltpb.HostInfo, error) {
	e := createEnv(rc.getAddr())
	defer e.conn.Close()
	defer e.cancel()
	resp, err := e.client.GetHost(e.ctx, &pb.GetHostRequest{Node: h})
	if err != nil {
		return nil, err
	}
	return resp.HostInfo, nil
}

func (rc *Reporter) GetUnit(u string) (*pb.GetUnitResponse, error) {
	e := createEnv(rc.getAddr())
	defer e.conn.Close()
	defer e.cancel()
	return e.client.GetUnit(e.ctx, &pb.GetUnitRequest{UnitName: u})
}

func (rc *Reporter) GetUnits() (*pb.GetUnitsResponse, error) {
	e := createEnv(rc.getAddr())
	defer e.conn.Close()
	defer e.cancel()
	return e.client.GetUnits(e.ctx, &pb.GetUnitsRequest{})
}

func (rc *Reporter) GetHosts() (HostsReader, error) {
	e := createEnv(rc.getAddr())
	stream, err := e.client.GetAllHosts(context.Background(), &pb.GetAllHostsRequest{})
	if err != nil {
		return nil, err
	}
	r := &hostsReader{
		done:   make(chan bool),
		conn:   e.conn,
		cancel: e.cancel,
		stream: stream,
	}
	return r, nil
}

func (rc *Reporter) GetHostsCount() (int, error) {
	e := createEnv(rc.getAddr())
	total, err := e.client.GetHostsCount(context.Background(), &pb.GetHostsCountRequest{})
	if err != nil {
		return 0, err
	}
	return int(total.Total), nil
}

type HostsReader interface {
	Read() (*yasaltpb.HostInfo, bool, error)
	io.Closer
}

type hostsReader struct {
	done      chan bool
	conn      *grpc.ClientConn
	cancel    context.CancelFunc
	stream    pb.Reporter_GetAllHostsClient
	p         int
	preloaded []*yasaltpb.HostInfo
}

func (r *hostsReader) Read() (*yasaltpb.HostInfo, bool, error) {
	if r.p >= len(r.preloaded) {
		r.p = 0
		resp, err := r.stream.Recv()
		if err == io.EOF {
			close(r.done)
			return nil, false, nil
		}
		if err != nil {
			return nil, false, err
		}
		r.preloaded = resp.Infos
	}
	info := r.preloaded[r.p]
	r.p++
	return info, true, nil
}

func (r *hostsReader) Close() error {
	r.cancel()
	return r.conn.Close()
}
