package server

import (
	"context"
	"fmt"
	"log"
	"time"

	"github.com/golang/protobuf/ptypes"
	"google.golang.org/protobuf/types/known/timestamppb"

	"a.yandex-team.ru/infra/hmserver/pkg/randutil"
	"a.yandex-team.ru/infra/hmserver/pkg/reporter/storage"
	"a.yandex-team.ru/infra/hmserver/pkg/reporter/types"
	"a.yandex-team.ru/infra/hmserver/pkg/yasmclient"
	yasaltpb "a.yandex-team.ru/infra/hostctl/proto"

	"a.yandex-team.ru/infra/hmserver/pkg/metrics"
	pb "a.yandex-team.ru/infra/hmserver/proto"
)

func CreateManager(hostsStorage storage.Hosts, unitsStorage storage.Units, heartbeatsStorage storage.Heartbeats, l *log.Logger) *Manager {
	metric := func(tier string) *metrics.Histogram {
		return metrics.NewHistogram(fmt.Sprintf("hm-report-%s_dhhh", tier), []float64{.001, .002, .004, .008, .016, .032, .064, .128, .256, .512, 1.024, 2.048, 4.096, 8.192})
	}
	mtrcs := &managerMetrics{
		handleReportsProcessingTime:    metric("handle-reports"),
		getReportsProcessingTime:       metric("get-reports"),
		getUnitsProcessingTime:         metric("get-units"),
		getHostsProcessingTime:         metric("get-hosts"),
		getHostProcessingTime:          metric("get-host"),
		getUnitVersionsProcessingTime:  metric("unit-versions"),
		getUnitStagesProcessingTime:    metric("unit-stages"),
		getUnitReadyProcessingTime:     metric("unit-ready"),
		getUnitPendingProcessingTime:   metric("unit-pending"),
		removeOldReportsProcessingTime: metric("remove-old-reports"),
		getHeartbeatsProcessingTime:    metric("get-heartbeats"),
	}
	menv := &managerEnv{
		unitsStorage:     unitsStorage,
		hostsStorage:     hostsStorage,
		heartbeatStorage: heartbeatsStorage,
		l:                l,
		metrics:          mtrcs,
	}
	return &Manager{
		managerEnv:        menv,
		reportManager:     &reportManager{managerEnv: menv, Enforcer: randutil.NewEnforcer()},
		hostManager:       &hostManager{managerEnv: menv},
		unitManager:       &unitManager{managerEnv: menv},
		statusManager:     &statusManager{managerEnv: menv},
		heartbeatsManager: &heartbeatsManager{managerEnv: menv},
	}
}

type Manager struct {
	*reportManager
	*hostManager
	*unitManager
	*statusManager
	*managerEnv
	*heartbeatsManager
}

type managerEnv struct {
	unitsStorage     storage.Units
	hostsStorage     storage.Hosts
	heartbeatStorage storage.Heartbeats
	l                *log.Logger
	metrics          *managerMetrics
}

type managerMetrics struct {
	handleReportsProcessingTime    *metrics.Histogram
	getReportsProcessingTime       *metrics.Histogram
	getUnitsProcessingTime         *metrics.Histogram
	getHostsProcessingTime         *metrics.Histogram
	getHostProcessingTime          *metrics.Histogram
	getUnitVersionsProcessingTime  *metrics.Histogram
	getUnitStagesProcessingTime    *metrics.Histogram
	getUnitReadyProcessingTime     *metrics.Histogram
	getUnitPendingProcessingTime   *metrics.Histogram
	removeOldReportsProcessingTime *metrics.Histogram
	getHeartbeatsProcessingTime    *metrics.Histogram
}

func (m *managerMetrics) ToPush() []yasmclient.YasmValue {
	return []yasmclient.YasmValue{
		m.handleReportsProcessingTime.FmtPush(),
		m.getReportsProcessingTime.FmtPush(),
		m.getUnitsProcessingTime.FmtPush(),
		m.getHostsProcessingTime.FmtPush(),
		m.getHostProcessingTime.FmtPush(),
		m.getUnitVersionsProcessingTime.FmtPush(),
		m.getUnitStagesProcessingTime.FmtPush(),
		m.getUnitReadyProcessingTime.FmtPush(),
		m.getUnitPendingProcessingTime.FmtPush(),
		m.removeOldReportsProcessingTime.FmtPush(),
		m.getHeartbeatsProcessingTime.FmtPush(),
	}
}

type reportManager struct {
	*managerEnv
	*randutil.Enforcer
}

// HandleReport
//
// report consist of units and host_include
// we have 3 table for storing this
//  * hosts (report time, units digests, host_include digest, host_include fields)
//  * units ('host -> []unit' matching)
//
// to decrease write operations rate we can
//   * update report_time
//   * for each report request get host_include, units digests
//   * with digests check that we has/not changes
//   * if host_include has changes -> update hosts table
//   * if units has changes -> update units table
//   * update digest if we had changes
func (m *reportManager) HandleReport(ctx context.Context, units []*pb.Unit, host *yasaltpb.HostInfo, unitsTSProto *timestamppb.Timestamp) error {
	start := time.Now()
	defer func() {
		//fmt.Println(float64(time.Since(start).Milliseconds()) / 1000)
		m.metrics.handleReportsProcessingTime.Observe(float64(time.Since(start).Milliseconds()) / 1000)
	}()
	if host == nil {
		return fmt.Errorf("failed to handle reports with empty host include")
	}
	reportTime := time.Now()
	hostTS, err := ptypes.Timestamp(host.Mtime)
	if err != nil {
		return err
	}
	hostTS = hostTS.Truncate(time.Second)
	unitsTS, err := ptypes.Timestamp(unitsTSProto)
	if err != nil {
		return err
	}
	unitsTS = unitsTS.Truncate(time.Second)
	oldHostTS, oldUnitsTS, err := m.hostsStorage.UpdateReportTime(ctx, host.Hostname, reportTime)
	if err != nil {
		m.l.Println("Failed to update report time", err)
		return err
	}
	// adding eps in case we have different accuracy of timestamps
	unitsChanged, hostChanged := unitsTS.After(oldUnitsTS), hostTS.After(oldHostTS)
	// update sometimes to ensure what reports will converge
	forceUpdate := m.Enforcer.Force()
	unitsChanged = forceUpdate || unitsChanged
	hostChanged = forceUpdate || hostChanged
	// update meta if we had some changes
	if unitsChanged || hostChanged {
		err = m.hostsStorage.UpsertMeta(ctx, host.Hostname, reportTime, types.HostTS(hostTS), types.UnitsTS(unitsTS))
		if err != nil {
			m.l.Println("Failed to upsert meta", err)
			return err
		}
	}
	// update units reports only if we have changes
	if unitsChanged {
		if err := m.unitsStorage.UpdateHostReports(ctx, host.Hostname, units); err != nil {
			m.l.Println("Failed to update units records", err)
			return err
		}
	}
	// upsert host include reports only if we have changes
	if hostChanged {
		if err = m.hostsStorage.UpdateHost(ctx, host); err != nil {
			m.l.Println("Failed to update hosts record", err)
			return err
		}
	}
	return nil
}

type hostManager struct {
	*managerEnv
}

func (m *Manager) GetHostReports(ctx context.Context, hostname types.Node, unit types.Unit, stage types.Stage, version types.Version, ready []pb.Status, pending []pb.Status, limit, offset int32) ([]*pb.Unit, error) {
	start := time.Now()
	defer func() {
		m.metrics.getReportsProcessingTime.Observe(float64(time.Since(start).Milliseconds()) / 1000)
	}()
	reports, err := m.unitsStorage.GetReports(ctx, hostname, unit, stage, version, ready, pending, limit, offset)
	if err != nil {
		m.l.Println(err.Error())
		return nil, err
	}
	return reports, nil
}

func (m *Manager) GetUnits(ctx context.Context) (map[string][]string, error) {
	start := time.Now()
	defer func() { m.metrics.getUnitsProcessingTime.Observe(float64(time.Since(start).Milliseconds()) / 1000) }()
	reports, err := m.unitsStorage.GetUnits(ctx)
	if err != nil {
		m.l.Println(err.Error())
		return nil, err
	}
	return reports, nil
}

func (m *Manager) GetHost(ctx context.Context, node types.Node) (*yasaltpb.HostInfo, error) {
	start := time.Now()
	defer func() { m.metrics.getHostProcessingTime.Observe(float64(time.Since(start).Milliseconds()) / 1000) }()
	host, err := m.hostsStorage.GetHost(ctx, string(node))
	if err != nil {
		m.l.Println(err.Error())
		return nil, err
	}
	return host, nil
}

func (m *Manager) GetHosts(ctx context.Context, include *yasaltpb.HostInfo) ([]string, error) {
	start := time.Now()
	defer func() { m.metrics.getHostsProcessingTime.Observe(float64(time.Since(start).Milliseconds()) / 1000) }()
	hosts, err := m.hostsStorage.GetHosts(ctx, include, 100, 0)
	if err != nil {
		return nil, err
	}
	return hosts, nil
}

type unitManager struct {
	*managerEnv
}

func (m *unitManager) GetUnitVersions(ctx context.Context, name types.Unit) (map[types.Version]int, error) {
	start := time.Now()
	defer func() {
		m.metrics.getUnitVersionsProcessingTime.Observe(float64(time.Since(start).Milliseconds()) / 1000)
	}()
	versions, err := m.unitsStorage.GetUnitVersions(ctx, name)
	if err != nil {
		m.l.Println(err.Error())
		return nil, err
	}
	return versions, nil
}

func (m *unitManager) GetUnitStages(ctx context.Context, name types.Unit) (map[types.Stage]int, error) {
	start := time.Now()
	defer func() {
		m.metrics.getUnitStagesProcessingTime.Observe(float64(time.Since(start).Milliseconds()) / 1000)
	}()
	stages, err := m.unitsStorage.GetUnitStages(ctx, name)
	if err != nil {
		m.l.Println(err.Error())
		return nil, err
	}
	return stages, nil
}

type statusManager struct {
	*managerEnv
}

func (m *statusManager) GetUnitReady(ctx context.Context, name types.Unit) (map[pb.Status]int, error) {
	start := time.Now()
	defer func() { m.metrics.getUnitReadyProcessingTime.Observe(float64(time.Since(start).Milliseconds()) / 1000) }()
	status, err := m.unitsStorage.GetUnitReady(ctx, name)
	if err != nil {
		m.l.Println(err.Error())
		return nil, err
	}
	return status, nil
}

func (m *statusManager) GetUnitPending(ctx context.Context, name types.Unit) (map[pb.Status]int, error) {
	start := time.Now()
	defer func() {
		m.metrics.getUnitPendingProcessingTime.Observe(float64(time.Since(start).Milliseconds()) / 1000)
	}()
	status, err := m.unitsStorage.GetUnitPending(ctx, name)
	if err != nil {
		m.l.Println(err.Error())
		return nil, err
	}
	return status, nil
}

func (m *statusManager) GetHostsCursor(ctx context.Context) (*storage.HostsCursor, error) {
	return m.hostsStorage.StartCursor(ctx), nil
}

// to remove reports of not existing units we:
//   * on request we can check that unit disappear from report and remove unit record
//   * by timeout per host in RemoveReportsOlder func: host had not send reports 2 days.
//     * remove hosts record for host older than time.Now() - ttl returning hostnames
//     * remove units records by hostnames
func (m *statusManager) RemoveReportsOlder(ctx context.Context, ttl time.Duration, limit int) (int, error) {
	start := time.Now()
	defer m.metrics.removeOldReportsProcessingTime.Observe(float64(time.Since(start).Milliseconds()) / 1000)
	hostsOlder, err := m.hostsStorage.RemoveHosts(ctx, ttl, limit)
	if err != nil {
		m.l.Printf("Failed to remove hosts records: %s\n", err)
		return 0, err
	}
	if err := m.unitsStorage.RemoveByNodes(ctx, hostsOlder); err != nil {
		m.l.Printf("Failed to remove units records: %s\n", err)
		return 0, err
	}
	return len(hostsOlder), nil
}

type heartbeatsManager struct {
	*managerEnv
}

func (m *heartbeatsManager) GetHeartbeats(ctx context.Context, fqdns []string) ([]*pb.HostHeartbeat, error) {
	start := time.Now()
	defer m.metrics.getHeartbeatsProcessingTime.Observe(float64(time.Since(start).Milliseconds()) / 1000)
	return m.heartbeatStorage.GetHeartbeats(ctx, fqdns...)
}
