package server

import (
	"a.yandex-team.ru/infra/hmserver/pkg/api"
	cockroach "a.yandex-team.ru/infra/hmserver/pkg/cockroach/api"
	"a.yandex-team.ru/infra/hmserver/pkg/cockroach/monitoring"
	"a.yandex-team.ru/infra/hmserver/pkg/tick"
	"a.yandex-team.ru/infra/hmserver/pkg/yasmclient"
	"a.yandex-team.ru/library/go/units"
	"context"
	"fmt"
	"github.com/jmoiron/sqlx"
	"log"
	"net/http"
	"os"
	"os/signal"
	"syscall"
	"time"

	"github.com/go-chi/chi/v5/middleware"

	"a.yandex-team.ru/infra/hmserver/pkg/httpserver"
	"a.yandex-team.ru/infra/hmserver/pkg/reporter/storage"
)

type Reporter struct {
	l        *log.Logger
	m        *Manager
	addr     string
	grpcAddr string
	conf     *storage.Config
	debug    bool
}

const (
	recordsTTL    = units.Day
	gcLimit       = 100
	gcTimeout     = 1 * time.Minute
	gcMinInterval = 6 * time.Minute
	gcMaxInterval = 9 * time.Minute
)

func NewReporter(l *log.Logger, addr, grpcAddr string, dbConfig *storage.Config, debug bool) (*Reporter, error) {
	db, err := sqlx.Open("pgx", storage.ConnString(dbConfig.Addr, dbConfig.DBName, dbConfig.User, dbConfig.Password))
	if err != nil {
		return nil, err
	}
	db.SetMaxOpenConns(200)
	db.SetMaxIdleConns(dbConfig.MaxIdleConns)
	db.SetConnMaxLifetime(dbConfig.MaxConnLifetime)
	unitsStorage, err := storage.NewUnits(db)
	if err != nil {
		return nil, err
	}
	hostsStorage, err := storage.NewHosts(db)
	if err != nil {
		return nil, err
	}
	heartbeatsStorage := storage.NewHeartbeatStorage(db)
	m := CreateManager(hostsStorage, unitsStorage, heartbeatsStorage, l)
	app := &Reporter{
		l:        l,
		m:        m,
		addr:     addr,
		grpcAddr: grpcAddr,
		debug:    debug,
		conf:     dbConfig,
	}
	return app, nil
}

func (r *Reporter) Run() {
	r.l.Println("Starting hm-reporter-server...")
	ctx, cancel := context.WithCancel(context.Background())
	server := NewServer(r.l, r.m, r.grpcAddr)
	hs := httpserver.NewHTTP(r.addr)
	// http api
	unistatAPI := api.NewUnistat(r.l)
	unistatAPI.Register(hs.Bind)
	mux := hs.Mux()
	if r.debug {
		mux.Mount("/debug", middleware.Profiler())
	}
	server.Register(mux)
	server.RegisterUI(hs.Handle, hs.Bind)
	errCh := make(chan error)
	sigs := make(chan os.Signal, 1)
	signal.Notify(sigs, syscall.SIGINT, syscall.SIGTERM)
	go func() {
		<-sigs
		cancel()
	}()
	go func() {
		<-sigs
		cancel()
	}()
	go func(ctx context.Context) {
		err := server.RunGrpc(ctx)
		if err != nil {
			errCh <- fmt.Errorf("grpc: %w", err)
		}
	}(ctx)
	go func(ctx context.Context) {
		r.l.Printf("Starting HTTP server on %s...", hs.Addr())
		if err := hs.Run(ctx); err != http.ErrServerClosed {
			errCh <- fmt.Errorf("http server: %w", err)
		}
	}(ctx)
	go func(ctx context.Context) {
		p := tick.NewPeriodic(r.l, "gc", gcMinInterval, CreateGC(recordsTTL, gcTimeout, gcLimit, r.l, r.m))
		p.WithRandomDelta(gcMaxInterval - gcMinInterval)
		if err := p.Run(ctx); err != nil {
			errCh <- fmt.Errorf("gc: %w", err)
		}
	}(ctx)
	go func(ctx context.Context) {
		m, err := NewMonitoring(r.conf, r.m, r.l)
		if err != nil {
			errCh <- fmt.Errorf("monitoring: %w", err)
		}
		if err := tick.NewPeriodic(r.l, "monitoring", 30*time.Second, m).Run(ctx); err != nil {
			errCh <- fmt.Errorf("monitoring: %w", err)
		}
	}(ctx)
	select {
	case err := <-errCh:
		r.l.Fatal(err.Error())
	case <-ctx.Done():
		r.l.Println("Terminating...")
	}
	r.l.Println("Exiting...")
}

func RunReporter(l *log.Logger, addr, grpcAddr string, dbConfig *storage.Config, debug bool) error {
	app, err := NewReporter(l, addr, grpcAddr, dbConfig, debug)
	if err != nil {
		return err
	}
	app.Run()
	return nil
}

func NewMonitoring(conf *storage.Config, m *Manager, l *log.Logger) (*MonitoringManager, error) {
	c, err := cockroach.NewAPI(conf.AdminAddr)
	if err != nil {
		return nil, err
	}
	return &MonitoringManager{
		conf: conf,
		m:    m,
		l:    l,
		c:    c,
	}, nil
}

type MonitoringManager struct {
	conf *storage.Config
	m    *Manager
	l    *log.Logger
	c    *cockroach.API
}

func (m *MonitoringManager) Sync(ctx context.Context) error {
	cockroachMetrics := monitoring.CollectCockroachSignals(ctx, m.l, m.c, m.conf.User, m.conf.Password, m.conf.AdminAddr)
	managerMetrics := m.m.metrics.ToPush()
	c := yasmclient.NewLocal()
	for _, metric := range append(cockroachMetrics, managerMetrics...) {
		m.l.Printf("pushing %s=%v\n", metric.Name, metric.Value)
	}
	err := c.SendMetrics(context.Background(), []yasmclient.YasmMetrics{{
		Tags: map[string]string{
			"itype": "runtimecloud",
			"ctype": "production"},
		TTL:    30 * 60,
		Values: managerMetrics,
	}, {
		Tags: map[string]string{
			"itype": "runtimecloud",
			"ctype": "production"},
		TTL:    30 * 60,
		Values: cockroachMetrics,
	}})
	if err != nil {
		m.l.Printf("Failed to push monitoring: %s\n", err)
		return err
	}
	m.l.Println("Successfully pushed monitoring")
	return nil
}
