package storage

import (
	"testing"
	"time"

	"a.yandex-team.ru/infra/hmserver/pkg/reporter/types"
	pb "a.yandex-team.ru/infra/hmserver/proto"
	yasaltpb "a.yandex-team.ru/infra/hostctl/proto"
	"github.com/golang/protobuf/ptypes"
	"github.com/stretchr/testify/assert"
)

var (
	node  = "sas1-5166.search.yandex.net"
	name  = "yandex-hm-reporter"
	name2 = "juggler-agent-rtc"
)

func Test_getUnitsByNodeQuery(t *testing.T) {
	q, err := getUnitsByNodeQuery()
	if err != nil {
		t.Error(err)
	}
	query, args, err := q(node)
	if err != nil {
		t.Error(err)
	}
	assert.Equal(t, "SELECT * FROM units WHERE node = $1", query)
	assert.Equal(t, []interface{}{node}, args)
}

func Test_getUnitsQuery(t *testing.T) {
	q, err := getUnitsQuery()
	if err != nil {
		t.Error(err)
	}
	query, args, err := q()
	if err != nil {
		t.Error(err)
	}
	assert.Equal(t, "SELECT DISTINCT name, kind FROM units", query)
	assert.Len(t, args, 0)
}

func Test_getUnitVersionsQuery(t *testing.T) {
	q, err := getUnitVersionsQuery()
	if err != nil {
		t.Error(err)
	}
	query, args, err := q(name)
	if err != nil {
		t.Error(err)
	}
	assert.Equal(t, "SELECT count(*), version FROM units WHERE name = $1 GROUP BY version", query)
	assert.Equal(t, []interface{}{name}, args)
}

func Test_getUnitReadyQuery(t *testing.T) {
	q, err := getUnitReadyQuery()
	if err != nil {
		t.Error(err)
	}
	query, args, err := q(name)
	if err != nil {
		t.Error(err)
	}
	assert.Equal(t, "SELECT count(*), ready FROM units WHERE name = $1 GROUP BY ready", query)
	assert.Equal(t, []interface{}{name}, args)
}

func Test_getUnitPendingQuery(t *testing.T) {
	q, err := getUnitPendingQuery()
	if err != nil {
		t.Error(err)
	}
	query, args, err := q(name)
	if err != nil {
		t.Error(err)
	}
	assert.Equal(t, "SELECT count(*), pending FROM units WHERE name = $1 GROUP BY pending", query)
	assert.Equal(t, []interface{}{name}, args)
}

func Test_getUnitStagesQuery(t *testing.T) {
	q, err := getUnitStagesQuery()
	if err != nil {
		t.Error(err)
	}
	query, args, err := q(name)
	if err != nil {
		t.Error(err)
	}
	assert.Equal(t, "SELECT count(*), stage FROM units WHERE name = $1 GROUP BY stage", query)
	assert.Equal(t, []interface{}{name}, args)
}

func Test_removeByNodeNamesQuery(t *testing.T) {
	q, err := removeByNodeNamesQuery()
	if err != nil {
		t.Error(err)
	}
	query, args, err := q(node, []string{name, name2})
	if err != nil {
		t.Error(err)
	}
	assert.Equal(t, "DELETE FROM units WHERE (node = $1 AND name IN ($2,$3))", query)
	assert.Equal(t, []interface{}{node, name, name2}, args)
}

func Test_removeByNodesQuery(t *testing.T) {
	q, err := removeByNodesQuery()
	if err != nil {
		t.Error(err)
	}
	query, args, err := q([]string{node})
	if err != nil {
		t.Error(err)
	}
	assert.Equal(t, "DELETE FROM units WHERE node IN ($1)", query)
	assert.Equal(t, []interface{}{node}, args)
}

func Test_upsertUnitsQuery(t *testing.T) {
	q, err := upsertUnitsQuery()
	if err != nil {
		t.Error(err)
	}
	now := ptypes.TimestampNow()
	n, err := ptypes.Timestamp(now)
	if err != nil {
		t.Error(err)
	}
	units := []*pb.Unit{{
		Node:           node,
		Name:           name,
		Version:        "stable",
		Stage:          "1",
		Kind:           "SystemService",
		Ready:          0,
		Pending:        0,
		LastTransition: now,
	}, {
		Node:           node,
		Name:           name2,
		Version:        "prestable",
		Stage:          "2",
		Kind:           "PortoDaemon",
		Ready:          1,
		Pending:        1,
		LastTransition: now,
	}}
	query, args, err := q(units)
	if err != nil {
		t.Error(err)
	}
	assert.Equal(t, "UPSERT INTO units VALUES ($1,$2,$3,$4,$5,$6,$7,$8),($9,$10,$11,$12,$13,$14,$15,$16)", query)
	assert.Equal(t, []interface{}{
		node, name, "stable", "1", "SystemService", 0, 0, n,
		node, name2, "prestable", "2", "PortoDaemon", 1, 1, n}, args)
}

func Test_getHostQuery(t *testing.T) {
	q, err := getHostQuery()
	if err != nil {
		t.Error(err)
	}
	query, args, err := q(node)
	if err != nil {
		t.Error(err)
	}
	assert.Equal(t, "SELECT hostname, num, walle_project, walle_tags, net_switch, gencfg_groups, location, dc, kernel, cpu_model, mem_total_mib, host_ts, dc_queue, os_codename, os_arch FROM hosts WHERE hostname = $1", query)
	assert.Equal(t, []interface{}{node}, args)
}

func Test_updateHostQuery(t *testing.T) {
	q, err := updateHostQuery()
	if err != nil {
		t.Error(err)
	}
	i := &yasaltpb.HostInfo{
		Hostname:      node,
		Num:           10,
		WalleProject:  "rtc",
		WalleTags:     []string{"rtc"},
		NetSwitch:     "sas1-34s",
		GencfgGroups:  []string{"GROUP"},
		Location:      "sas",
		Dc:            "sas",
		KernelRelease: "4.19.119-30.2",
		CpuModel:      "AMD EPYC 7351 16-Core Processor",
		MemTotalMib:   257920,
		DcQueue:       "sas1.2.1",
		OsCodename:    "xenial",
		OsArch:        "amd64",
	}
	query, args, err := q(i)
	if err != nil {
		t.Error(err)
	}
	assert.Equal(t, "UPDATE hosts SET num = $1, walle_project = $2, walle_tags = $3, net_switch = $4, gencfg_groups = $5, location = $6, dc = $7, kernel = $8, cpu_model = $9, mem_total_mib = $10, dc_queue = $11, os_codename = $12, os_arch = $13 WHERE hostname = $14", query)
	assert.Equal(t, []interface{}{int32(10), "rtc", []string{"rtc"}, "sas1-34s", []string{"GROUP"}, "sas", "sas", "4.19.119-30.2", "AMD EPYC 7351 16-Core Processor", int32(257920), "sas1.2.1", "xenial", "amd64", "sas1-5166.search.yandex.net"}, args)
}

func Test_upsertMetaQuery(t *testing.T) {
	q, err := upsertMetaQuery()
	if err != nil {
		t.Error(err)
	}
	now := time.Now()
	query, args, err := q(node, types.UnitsTS(now.Add(time.Second)), types.HostTS(now.Add(time.Second*2)), now)
	if err != nil {
		t.Error(err)
	}
	assert.Equal(t, "UPSERT INTO hosts (hostname,units_ts,host_ts,report_time) VALUES ($1,$2,$3,$4)", query)
	assert.Len(t, args, 4)
	assert.IsType(t, args[0], node)
	assert.IsType(t, args[1], types.UnitsTS(now.Add(time.Second)))
	assert.IsType(t, args[2], types.HostTS(now.Add(time.Second*2)))
	assert.IsType(t, args[3], now)
}

func Test_updateReportTimeQuery(t *testing.T) {
	q, err := updateReportTimeQuery()
	if err != nil {
		t.Error(err)
	}
	now := time.Now()
	query, args, err := q(now, node)
	if err != nil {
		t.Error(err)
	}
	assert.Equal(t, "UPDATE hosts SET report_time = $1 WHERE hostname = $2 RETURNING host_ts, units_ts", query)
	assert.Equal(t, []interface{}{now, node}, args)
}

func Test_getHostsQuery(t *testing.T) {
	q, err := getHostsQuery()
	if err != nil {
		t.Error(err)
	}
	i := &yasaltpb.HostInfo{
		Hostname:      node,
		Num:           10,
		WalleProject:  "rtc",
		WalleTags:     []string{"rtc"},
		NetSwitch:     "sas1-34s",
		GencfgGroups:  []string{"GROUP"},
		Location:      "sas",
		Dc:            "sas",
		KernelRelease: "4.19.119-30.2",
		CpuModel:      "AMD EPYC 7351 16-Core Processor",
		MemTotalMib:   257920,
	}
	query, args, err := q(i, uint64(100), uint64(20))
	if err != nil {
		t.Error(err)
	}
	assert.Equal(t, "SELECT hostname FROM hosts WHERE hostname LIKE $1 AND num <= $2 AND walle_project = $3 AND walle_tags && array['rtc'] AND net_switch = $4 AND gencfg_groups && array['GROUP'] AND location = $5 AND dc = $6 AND kernel = $7 LIMIT 100 OFFSET 20", query)
	assert.Equal(t, []interface{}{i.Hostname + "%", i.Num, i.WalleProject, i.NetSwitch, i.Location, i.Dc, i.KernelRelease}, args)
}

func Test_removeHostsQuery(t *testing.T) {
	q, err := removeHostsQuery()
	if err != nil {
		t.Error(err)
	}
	query, args, err := q(time.Second, uint64(100))
	if err != nil {
		t.Error(err)
	}
	assert.Equal(t, "DELETE FROM hosts WHERE report_time < $1 LIMIT 100 RETURNING (hostname)", query)
	assert.Len(t, args, 1)
	assert.IsType(t, time.Time{}, args[0])
}

func Test_getHostsCursor(t *testing.T) {
	q, err := getHostsCursor()
	if err != nil {
		t.Error(err)
	}
	now := time.Now().Truncate(time.Second)
	query, _, err := q(uint64(100), uint64(100), now)
	if err != nil {
		t.Error(err)
	}
	assert.Equal(t, "SELECT hostname, num, walle_project, walle_tags, net_switch, gencfg_groups, location, dc, kernel, cpu_model, mem_total_mib, host_ts, dc_queue, os_codename, os_arch FROM hosts LIMIT 100 OFFSET 100", query)
}

func Test_hostsCountQuery(t *testing.T) {
	q, err := hostsCountQuery()
	if err != nil {
		t.Error(err)
	}
	now := time.Now().Truncate(time.Second)
	query, _, err := q(uint64(100), uint64(100), now)
	if err != nil {
		t.Error(err)
	}
	assert.Equal(t, "SELECT count(*) FROM hosts", query)
}

func Test_kernelVersionsQuery(t *testing.T) {
	q, err := kernelVersionsQuery()
	if err != nil {
		t.Error(err)
	}
	query, _, err := q()
	if err != nil {
		t.Error(err)
	}
	assert.Equal(t, "SELECT kernel, count(*) FROM hosts GROUP BY kernel", query)
}

func Test_getUnitsReportsQuery(t *testing.T) {
	q, err := getUnitsReportsQuery()
	if err != nil {
		t.Error(err)
	}
	u := &pb.Unit{
		Node:    node,
		Name:    name,
		Version: "1",
		Stage:   "stable",
	}
	ready := []pb.Status{pb.Status_TRUE}
	pending := []pb.Status{pb.Status_FALSE}
	query, args, err := q(u, ready, pending, uint64(100), uint64(20))
	if err != nil {
		t.Error(err)
	}
	assert.Equal(t, "SELECT * FROM units WHERE node = $1 AND name = $2 AND stage = $3 AND version = $4 AND ready IN ($5) AND pending IN ($6) LIMIT 100 OFFSET 20", query)
	assert.Equal(t, []interface{}{u.Node, u.Name, u.Stage, u.Version, pb.Status_TRUE, pb.Status_FALSE}, args)
}
