package storage

import (
	"context"
	"net"
	"strconv"
	"strings"

	"github.com/jmoiron/sqlx"
)

/*
Indexes stage_stored_index, version_stored_index, pending_stored_index, ready_stored_index
needed to improve performance ui on unit page with requests like
SELECT count(*), <stage, version, pending, ready> FROM units GROUP BY <stage, version, pending, ready>;

Index kernel_stored_index needed to improve performance ui on kernel page with requests like
SELECT count(*), kernel FROM hosts GROUP BY kernel;

Index name_kind_index needed to improve performance ui on units listing page with requests like
SELECT DISTINCT (name, kind) FROM units;
*/
var (
	hostsScheme = `CREATE TABLE IF NOT EXISTS hosts (
	hostname            TEXT NOT NULL PRIMARY KEY DEFAULT '',
	host_ts             TIMESTAMP,
	units_ts            TIMESTAMP,
	report_time         TIMESTAMP,
	num                 INT,
	walle_project       TEXT,
	walle_tags          VARCHAR[],
	net_switch          TEXT,
	gencfg_groups       VARCHAR[],
	location            TEXT,
	dc                  TEXT,
	kernel              TEXT,
	cpu_model           TEXT,
	mem_total_mib       INT,
    dc_queue			TEXT,
    os_codename			TEXT,
	os_arch				TEXT
);

CREATE INDEX IF NOT EXISTS kernel_stored_index on hosts (hostname) storing (kernel);`
	unitsScheme = `CREATE TABLE IF NOT EXISTS units (
	node 			TEXT NOT NULL DEFAULT '',
	name            TEXT NOT NULL DEFAULT '',
	version         TEXT DEFAULT '',
	stage           TEXT DEFAULT '',
	kind            TEXT DEFAULT '',
	ready           INT DEFAULT 1,
	pending         INT DEFAULT 1,
	last_transition TIMESTAMP,
	PRIMARY KEY (node, name)
);

CREATE INDEX IF NOT EXISTS stage_stored_index on units (name) storing (stage);
CREATE INDEX IF NOT EXISTS version_stored_index on units (name) storing (version);
CREATE INDEX IF NOT EXISTS pending_stored_index on units (name) storing (pending);
CREATE INDEX IF NOT EXISTS ready_stored_index on units (name) storing (ready);

CREATE INDEX IF NOT EXISTS name_kind_index on units (name, kind);`
)

// ConnString constructs PostgreSQL connection string
func ConnString(addr, dbname, user, password string) string {
	var connParams []string

	host, port, err := net.SplitHostPort(addr)
	if err == nil {
		connParams = append(connParams, "host="+host)
		connParams = append(connParams, "port="+port)
	} else {
		connParams = append(connParams, "host="+addr)
	}

	if dbname != "" {
		connParams = append(connParams, "dbname="+dbname)
	}

	if user != "" {
		connParams = append(connParams, "user="+user)
	}

	if password != "" {
		connParams = append(connParams, "password="+password)
	}

	connParams = append(connParams, "sslmode="+"require")

	return strings.Join(connParams, " ")
}

func exec(db *sqlx.DB, ctx context.Context, query string, args []interface{}) error {
	_, err := db.QueryxContext(ctx, ReplaceSQL(query, "?"), args...)
	return err
}

func ReplaceSQL(old, searchPattern string) string {
	tmpCount := strings.Count(old, searchPattern)
	for m := 1; m <= tmpCount; m++ {
		old = strings.Replace(old, searchPattern, "$"+strconv.Itoa(m), 1)
	}
	return old
}
