package storage

import (
	pb "a.yandex-team.ru/infra/hmserver/proto"
	"fmt"
	"reflect"
	"testing"
)

func Test_getUnitsToExecute(t *testing.T) {
	type args struct {
		oldUnits      []string
		incomingUnits []string
		changed       bool
	}
	tests := []struct {
		name         string
		args         args
		wantToUpsert []string
		wantToRemove []string
	}{{
		name: "new unit",
		args: args{
			oldUnits:      make([]string, 0),
			incomingUnits: []string{"new"},
		},
		wantToUpsert: []string{"new"},
		wantToRemove: make([]string, 0),
	}, {
		name: "to remove",
		args: args{
			oldUnits:      []string{"to_remove"},
			incomingUnits: make([]string, 0),
		},
		wantToUpsert: make([]string, 0),
		wantToRemove: []string{"to_remove"},
	}, {
		name: "upsert/remove",
		args: args{
			oldUnits:      []string{"to_remove"},
			incomingUnits: []string{"inc"},
		},
		wantToUpsert: []string{"inc"},
		wantToRemove: []string{"to_remove"},
	}, {
		name: "upsert/remove/notchanged",
		args: args{
			oldUnits:      []string{"not_changed", "to_remove"},
			incomingUnits: []string{"inc", "not_changed"},
		},
		wantToUpsert: []string{"inc"},
		wantToRemove: []string{"to_remove"},
	}, {
		name: "upsert/notchanged",
		args: args{
			oldUnits:      []string{"not_changed"},
			incomingUnits: []string{"inc", "not_changed"},
		},
		wantToUpsert: []string{"inc"},
		wantToRemove: make([]string, 0),
	}, {
		name: "remove/notchanged",
		args: args{
			oldUnits:      []string{"not_changed", "to_remove"},
			incomingUnits: []string{"not_changed"},
		},
		wantToUpsert: make([]string, 0),
		wantToRemove: []string{"to_remove"},
	}, {
		name: "not changed",
		args: args{
			oldUnits:      []string{"old"},
			incomingUnits: []string{"old"},
		},
		wantToUpsert: make([]string, 0),
		wantToRemove: make([]string, 0),
	}, {
		name: "changed",
		args: args{
			oldUnits:      []string{"changed"},
			incomingUnits: []string{"changed"},
			changed:       true,
		},
		wantToUpsert: []string{"changed"},
		wantToRemove: make([]string, 0),
	}}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			old, inc := namesToUnits(tt.args.oldUnits), namesToUnits(tt.args.incomingUnits)
			if tt.args.changed {
				old, inc = namesToDifferentUnits(tt.args.oldUnits, tt.args.incomingUnits)
			}
			gotToUpsert, gotToRemove := getUnitsToExecute(old, inc)
			// set empty stage to compare only names
			for _, unit := range gotToUpsert {
				unit.Stage = ""
			}
			for _, unit := range gotToRemove {
				unit.Stage = ""
			}
			if !reflect.DeepEqual(gotToUpsert, namesToUnits(tt.wantToUpsert)) {
				t.Errorf("getUnitsToExecute() gotToUpsert = %v, want %v", gotToUpsert, namesToUnits(tt.wantToUpsert))
			}
			if !reflect.DeepEqual(gotToRemove, namesToUnits(tt.wantToRemove)) {
				t.Errorf("getUnitsToExecute() gotToRemove = %v, want %v", gotToRemove, namesToUnits(tt.wantToRemove))
			}
		})
	}
}

func namesToUnits(names []string) []*pb.Unit {
	units := make([]*pb.Unit, len(names))
	for i, name := range names {
		units[i] = &pb.Unit{Name: name}
	}
	return units
}

func namesToDifferentUnits(a, b []string) ([]*pb.Unit, []*pb.Unit) {
	unitsA := make([]*pb.Unit, len(a))
	unitsB := make([]*pb.Unit, len(a))
	j := 0
	for i, name := range a {
		unitsA[i] = &pb.Unit{Name: name, Stage: fmt.Sprintf("%d", j)}
		j++
	}
	for i, name := range b {
		unitsB[i] = &pb.Unit{Name: name, Stage: fmt.Sprintf("%d", j)}
		j++
	}
	return unitsA, unitsB
}
