package squtil

import (
	"bytes"
	"errors"
	"fmt"
	"io"
	"strings"

	sq "github.com/Masterminds/squirrel"
)

func Upsert(into string) UpsertBuilder {
	return UpsertBuilder{
		&upsertData{Into: into, Suffixes: make([]string, 0)},
	}
}

type upsertData struct {
	Options  []string
	Into     string
	Columns  []string
	Values   [][]interface{}
	Suffixes []string
}

type suffix struct {
	sql  string
	args []interface{}
}

func (d *upsertData) ToSQL() (sqlStr string, args []interface{}, err error) {
	if len(d.Into) == 0 {
		err = errors.New("upsert statements must specify a table")
		return
	}
	if len(d.Values) == 0 {
		err = errors.New("upsert statements must have values clause")
		return
	}
	sql := &bytes.Buffer{}
	sql.WriteString("UPSERT ")
	sql.WriteString("INTO ")
	sql.WriteString(d.Into)
	sql.WriteString(" ")
	if len(d.Columns) > 0 {
		sql.WriteString("(")
		sql.WriteString(strings.Join(d.Columns, ","))
		sql.WriteString(") ")
	}
	args, err = d.appendValuesToSQL(sql, args)
	for _, s := range d.Suffixes {
		sql.WriteString(" ")
		sql.WriteString(s)
	}
	sqlStr = sql.String()
	return
}

func (d *upsertData) appendValuesToSQL(w io.Writer, args []interface{}) ([]interface{}, error) {
	if len(d.Values) == 0 {
		return args, errors.New("values for upsert statements are not set")
	}
	_, _ = io.WriteString(w, "VALUES ")
	valuesStrings := make([]string, len(d.Values))
	for r, row := range d.Values {
		valueStrings := make([]string, len(row))
		for v, val := range row {
			if vs, ok := val.(sq.Sqlizer); ok {
				vsql, vargs, err := vs.ToSql()
				if err != nil {
					return nil, err
				}
				valueStrings[v] = vsql
				args = append(args, vargs...)
			} else {
				valueStrings[v] = "?"
				args = append(args, val)
			}
		}
		valuesStrings[r] = fmt.Sprintf("(%s)", strings.Join(valueStrings, ","))
	}
	_, _ = io.WriteString(w, strings.Join(valuesStrings, ","))
	return args, nil
}

// UpsertBuilder builds SQL INSERT statements.
type UpsertBuilder struct {
	*upsertData
}

// Format methods

// SQL methods

// ToSQL builds the query into a SQL string and bound args.
func (b UpsertBuilder) ToSQL() (string, []interface{}, error) {
	return b.upsertData.ToSQL()
}

// Columns adds upsert columns to the query.
func (b UpsertBuilder) Columns(columns ...string) UpsertBuilder {
	b.upsertData.Columns = columns
	return b
}

// Values adds a single row's values to the query.
func (b UpsertBuilder) Values(values ...interface{}) UpsertBuilder {
	b.upsertData.Values = append(b.upsertData.Values, values)
	return b
}

func (b UpsertBuilder) Suffix(sql string) UpsertBuilder {
	b.upsertData.Suffixes = append(b.upsertData.Suffixes, sql)
	return b
}
