/* Synchronizes salt repository from bb.yandex-team.ru */
package sync

import (
	"archive/zip"
	"bytes"
	"context"
	"errors"
	"fmt"
	"io"
	"log"
	"sync/atomic"
	"time"
	"unsafe"

	"a.yandex-team.ru/infra/hmserver/pkg/bitbucket"
	"a.yandex-team.ru/infra/hmserver/pkg/types"
)

type BBSync struct {
	l  *log.Logger
	bb bitbucket.Bitbucket
	z  *types.SaltZip
}

func NewBB(l *log.Logger, bb bitbucket.Bitbucket) *BBSync {
	return &BBSync{
		l:  l,
		bb: bb,
	}
}

/* repackZip creates new archive without test files */
func repackZip(buf []byte) ([]byte, error) {
	in, err := zip.NewReader(bytes.NewReader(buf), int64(len(buf)))
	if err != nil {
		return nil, err
	}
	if len(in.File) == 0 {
		return nil, errors.New("empty zip archive")
	}
	b := &bytes.Buffer{}
	b.Grow(len(buf))
	out := zip.NewWriter(b)
	for _, f := range in.File {
		if blacklisted(f.Name) {
			continue
		}
		r, err := f.Open()
		if err != nil {
			return nil, fmt.Errorf("failed to open '%s': %s", f.Name, err)
		}
		w, err := out.Create(f.Name)
		if err != nil {
			return nil, fmt.Errorf("failed to add '%s': %s", f.Name, err)
		}
		if _, err := io.Copy(w, r); err != nil {
			return nil, fmt.Errorf("failed to copy '%s': %s", f.Name, err)
		}
		if err := r.Close(); err != nil {
			return nil, fmt.Errorf("failed to close zip: %s", err)
		}
	}
	if err := out.Close(); err != nil {
		return nil, fmt.Errorf("failed to close zip: %s", err)
	}
	return b.Bytes(), nil
}

func (bbs *BBSync) setZIP(z *types.SaltZip) {
	addr := (*unsafe.Pointer)(unsafe.Pointer(&bbs.z))
	atomic.StorePointer(addr, unsafe.Pointer(z))
}

func (bbs *BBSync) Sync(ctx context.Context) error {
	ci, err := bbs.bb.GetLastCommit(ctx)
	if err != nil {
		return fmt.Errorf("failed to sync: %s", err)
	}
	commitID := ""
	if bbs.z != nil {
		commitID = bbs.z.CommitID
	}
	if ci.ID != commitID {
		bbs.l.Printf("Commit changed: current=%s new=%s", commitID, ci.ID)
		bbs.l.Println("Downloading zip archive...")
		buf, err := bbs.bb.ZipArchive(ctx, ci.ID)
		if err != nil {
			// Don't update last commit, thus will retry
			return fmt.Errorf("failed to download: %s", err)
		}
		// Throw away some unused files *and* meanwhile - perform some basic validation,
		// that this is at least zip with some files in it.
		buf, err = repackZip(buf)
		if err != nil {
			return fmt.Errorf("failed to repack archive: %s", err)
		}
		z := &types.SaltZip{
			Content:   buf,
			CommitID:  ci.ID,
			Timestamp: time.Unix(ci.CommitterTimestamp/1000, 0),
			Message:   ci.Message,
		}
		bbs.setZIP(z)
		bbs.l.Println("Successfully downloaded zip for", ci.ID)
	} else {
		bbs.l.Println("Commit not changed, current:", bbs.z.CommitID)
	}
	return nil
}

func (bbs *BBSync) GetZip() (*types.SaltZip, error) {
	if bbs.z == nil {
		return nil, errors.New("no zip yet")
	}
	return bbs.z, nil
}
