package sync

import (
	"archive/zip"
	"bytes"
	"context"
	"errors"
	"io/ioutil"
	"log"
	"testing"
	"time"

	"a.yandex-team.ru/infra/hmserver/pkg/bitbucket"
	"a.yandex-team.ru/infra/hmserver/pkg/types"
)

func TestBBSync_GetZip(t *testing.T) {
	bbs := BBSync{}
	if _, err := bbs.GetZip(); err == nil {
		t.Error("Empty bbs didn't return err")
	}
	z := &types.SaltZip{}
	bbs.z = z
	got, err := bbs.GetZip()
	if err != nil {
		t.Error("BBS failed to return zip:", err)
	} else if got != z {
		t.Error("BBS returned invalid zip")
	}
}

func TestBBSync_Sync(t *testing.T) {
	f := bitbucket.FakeBB{}
	bbs := NewBB(log.New(ioutil.Discard, "", 0), &f)
	ctx := context.Background()
	// Fail case on GetLastCommit
	f.GetLastCommitFun = func(ctx context.Context) (info *bitbucket.CommitInfo, e error) {
		return nil, errors.New("failed to test")
	}
	if err := bbs.Sync(ctx); err == nil {
		t.Fatal("Sync not failed")
	}
	// Fail case on ZipArchive
	ci := &bitbucket.CommitInfo{
		ID:                 "7912362123123",
		CommitterTimestamp: time.Now().Unix() * 1000,
		Message:            "Test commit",
	}
	f.GetLastCommitFun = func(ctx context.Context) (info *bitbucket.CommitInfo, e error) {
		return ci, nil
	}
	f.ZipArchiveFun = func(ctx context.Context, at string) (bytes []byte, e error) {
		if at != ci.ID {
			t.Fatal("Requested zip for unexpected commit:", at)
		}
		return nil, errors.New("failed to get zip")
	}
	if err := bbs.Sync(ctx); err == nil {
		t.Fatal("Sync not failed")
	}
	buf := []byte("This is zip file, trust me!")
	// Invalid zip content
	f.ZipArchiveFun = func(ctx context.Context, at string) (bytes []byte, e error) {
		if at != ci.ID {
			t.Fatal("Requested zip for unexpected commit:", at)
		}
		return buf, nil
	}
	if err := bbs.Sync(ctx); err == nil {
		t.Fatal("Sync failed:", err)
	}
	// Good case with proper zip file
	b := bytes.Buffer{}
	z := zip.NewWriter(&b)
	zf, err := z.Create("search_runtime/top.sls")
	if err != nil {
		t.Fatal(err)
	}
	if _, err := zf.Write([]byte("print('OK')")); err != nil {
		t.Fatal(err)
	}
	if err := z.Close(); err != nil {
		t.Fatal(err)
	}
	f.ZipArchiveFun = func(ctx context.Context, at string) (bytes []byte, e error) {
		if at != ci.ID {
			t.Fatal("Requested zip for unexpected commit:", at)
		}
		return b.Bytes(), nil
	}
	if err := bbs.Sync(ctx); err != nil {
		t.Fatal("Sync failed:", err)
	}
	zInfo, err := bbs.GetZip()
	if err != nil {
		t.Fatal("GetZip() failed:", err)
	}
	if zInfo.CommitID != ci.ID {
		t.Errorf("Invalid commit id: %s != %s", zInfo.CommitID, ci.ID)
	}
	if !bytes.Equal(zInfo.Content, b.Bytes()) {
		t.Errorf("Invalid zip content")
	}
	if ts := zInfo.Timestamp.Unix() * 1000; ts != ci.CommitterTimestamp {
		t.Errorf("Invalid timestamps: %d != %d", ts, ci.CommitterTimestamp)
	}
}

func Test_repackFile(t *testing.T) {
	// Test empty case
	if _, err := repackZip(nil); err == nil {
		t.Error("No error with empty buffer")
	}
	// Test with remove tests
	b := bytes.Buffer{}
	z := zip.NewWriter(&b)
	f, err := z.Create("tests/test_something.py")
	if err != nil {
		t.Fatal(err)
	}
	if _, err := f.Write([]byte("print('OK')")); err != nil {
		t.Fatal(err)
	}
	f, err = z.Create("search_runtime/top.sls")
	if err != nil {
		t.Fatal(err)
	}
	if _, err := f.Write([]byte("print('OK')")); err != nil {
		t.Fatal(err)
	}
	if err := z.Close(); err != nil {
		t.Fatal(err)
	}
	buf, err := repackZip(b.Bytes())
	if err != nil {
		t.Fatal(err)
	}
	// Check that zip file is correct and has one file
	r, err := zip.NewReader(bytes.NewReader(buf), int64(len(buf)))
	if err != nil {
		t.Fatal(err)
	}
	if l := len(r.File); l != 1 {
		t.Fatal("Unexpected number of files in zip: ", l)
	}
	if name := r.File[0].Name; name != "search_runtime/top.sls" {
		t.Error("Unexpected file name: ", name)
	}
}
