/* Synchronizes salt repository from file by polling */
package sync

import (
	"archive/zip"
	"bytes"
	"context"
	"errors"
	"fmt"
	"io"
	"os"
	"path/filepath"
	"strings"
	"sync/atomic"
	"time"
	"unsafe"

	"a.yandex-team.ru/infra/hmserver/pkg/types"
)

type DirectorySync struct {
	path string
	z    *types.SaltZip
}

func NewDirectory(path string) *DirectorySync {
	return &DirectorySync{
		path: path,
	}
}

func (ds *DirectorySync) setZIP(z *types.SaltZip) {
	addr := (*unsafe.Pointer)(unsafe.Pointer(&ds.z))
	atomic.StorePointer(addr, unsafe.Pointer(z))
}

func (ds *DirectorySync) Sync(ctx context.Context) error {
	b := &bytes.Buffer{}
	z := zip.NewWriter(b)
	mtime := time.Time{} // Latest mtime of a file in directory
	err := filepath.Walk(ds.path, func(path string, info os.FileInfo, err error) error {
		if info.IsDir() {
			return nil
		}
		if err != nil {
			return err
		}
		name := strings.TrimPrefix(path, ds.path)
		// Skip tests and .git directories
		if strings.HasPrefix(name, "tests") || strings.HasPrefix(name, ".git") {
			return nil
		}
		if info.ModTime().After(mtime) {
			mtime = info.ModTime()
		}
		dst, err := z.Create(name)
		if err != nil {
			return err
		}
		src, err := os.Open(path)
		if err != nil {
			return err
		}
		_, err = io.Copy(dst, src)
		if err != nil {
			return err
		}
		return nil
	})
	if err != nil {
		return fmt.Errorf("dir walk failed: %s", err)
	}
	if err := z.Close(); err != nil {
		return fmt.Errorf("zip close failed: %s", err)
	}
	ret := &types.SaltZip{
		Timestamp: mtime,
		Content:   b.Bytes(),
		CommitID:  "<dir-sync-no-info>",
		Message:   "Gathered from directory",
	}
	ds.setZIP(ret)
	return nil
}

func (ds *DirectorySync) GetZip() (*types.SaltZip, error) {
	if ds.z == nil {
		return nil, errors.New("no zip yet")
	}
	return ds.z, nil
}
