/* Synchronizes salt repository from file by polling */
package sync

import (
	"bytes"
	"context"
	"errors"
	"fmt"
	"os"
	"sync/atomic"
	"time"
	"unsafe"

	"a.yandex-team.ru/infra/hmserver/pkg/types"
)

type FileSync struct {
	path string
	z    *types.SaltZip
}

func NewFile(path string) *FileSync {
	return &FileSync{
		path: path,
	}
}

func (fs *FileSync) setZIP(z *types.SaltZip) {
	addr := (*unsafe.Pointer)(unsafe.Pointer(&fs.z))
	atomic.StorePointer(addr, unsafe.Pointer(z))
}

func (fs *FileSync) Sync(ctx context.Context) error {
	f, err := os.Open(fs.path)
	if err != nil {
		return err
	}
	stat, err := f.Stat()
	if err != nil {
		return fmt.Errorf("fstat(%s) failed: %s", fs.path, err)
	}
	if stat.Size() > 50*1000*1000 {
		return fmt.Errorf("file '%s' is too big: %d", fs.path, stat.Size())
	}
	b := bytes.Buffer{}
	b.Grow(int(stat.Size()))
	if _, err := b.ReadFrom(f); err != nil {
		return fmt.Errorf("read '%s' failed: %s", fs.path, err)
	}
	zip := &types.SaltZip{
		Timestamp: time.Unix(int64(stat.ModTime().Second()), 0),
		Content:   b.Bytes(),
		CommitID:  "<file-sync-no-info>",
		Message:   "Gathered from file",
	}
	fs.setZIP(zip)
	return nil
}

func (fs *FileSync) GetZip() (*types.SaltZip, error) {
	if fs.z == nil {
		return nil, errors.New("no zip yet")
	}
	return fs.z, nil
}
