/*
Synchronizes multiple hostman/salt repository from bb.yandex-team.ru.

To be fully backward compatible, we merge all repository zip files into one,
prefixing non `saltstack` named, thus we guarantee conflict free merge.

If one repository fails to sync - all others **ARE SKIPPED**, thus we can have
same simple caching protocol based on commit timestamps.

Drawback - if we add new repository with older timestamp - only hosts without cache
will see it (eventually all of them, but). We assume this trade off is tolerable.
*/
package sync

import (
	"archive/zip"
	"bytes"
	"context"
	"errors"
	"fmt"
	"io"
	"log"
	"path/filepath"

	"sync/atomic"
	"time"
	"unsafe"

	"a.yandex-team.ru/infra/hmserver/pkg/bitbucket"
	"a.yandex-team.ru/infra/hmserver/pkg/types"
)

type (
	MultiBBsync struct {
		l        *log.Logger
		remotes  map[string]bitbucket.Bitbucket // Clients to remote repositories
		statuses map[string]*types.SaltZip      // Remote timestamps
		z        *types.SaltZip                 // Resulting zip archive with metadata
	}
)

func NewMultiBB(l *log.Logger, remotes map[string]bitbucket.Bitbucket) *MultiBBsync {
	statuses := make(map[string]*types.SaltZip, len(remotes))
	return &MultiBBsync{
		l:        l,
		remotes:  remotes,
		statuses: statuses,
	}
}

func merge(remotes map[string]*types.SaltZip) (*types.SaltZip, error) {
	outBuf := &bytes.Buffer{}
	outBuf.Grow(512 * 1024)
	out := zip.NewWriter(outBuf)
	rv := &types.SaltZip{}
	for name, sz := range remotes {
		in, err := zip.NewReader(bytes.NewReader(sz.Content), int64(len(sz.Content)))
		if err != nil {
			return nil, err
		}
		// Update metadata, by taking latest commit and ts
		if rv.Timestamp.Before(sz.Timestamp) {
			rv.Timestamp = sz.Timestamp
			rv.Message = sz.Message
			rv.CommitID = sz.CommitID
		}
		prefix := ""
		// Old common repo does not have a prefix
		if name != "saltstack" {
			prefix = name
		}
		for _, f := range in.File {
			if blacklisted(f.Name) {
				continue
			}
			if f.FileInfo().IsDir() {
				continue
			}
			r, err := f.Open()
			if err != nil {
				return nil, fmt.Errorf("failed to open '%s': %s", f.Name, err)
			}
			fName := filepath.Join(prefix, f.Name)
			w, err := out.Create(fName)
			if err != nil {
				return nil, fmt.Errorf("failed to add '%s': %s", fName, err)
			}
			if _, err := io.Copy(w, r); err != nil {
				return nil, fmt.Errorf("failed to copy '%s': %s", fName, err)
			}
			if err := r.Close(); err != nil {
				return nil, fmt.Errorf("failed to close zip: %s", err)
			}
		}
	}
	if err := out.Close(); err != nil {
		return nil, fmt.Errorf("failed to close zip: %s", err)
	}
	rv.Content = outBuf.Bytes()
	return rv, nil
}

func (mbbs *MultiBBsync) setZIP(z *types.SaltZip) {
	addr := (*unsafe.Pointer)(unsafe.Pointer(&mbbs.z))
	atomic.StorePointer(addr, unsafe.Pointer(z))
}

func (mbbs *MultiBBsync) Sync(ctx context.Context) error {
	// For every repo in remotes, grab new zip
	for name, bb := range mbbs.remotes {
		status, ok := mbbs.statuses[name]
		if !ok {
			status = &types.SaltZip{}
			mbbs.statuses[name] = status
		}
		ci, err := bb.GetLastCommit(ctx)
		if err != nil {
			return fmt.Errorf("failed to sync: %s", err)
		}
		if status.CommitID == ci.ID {
			mbbs.l.Printf("Repo '%s': commit '%s' has not changed", name, ci.ID[:11])
			continue
		}
		commitID := ci.ID
		mbbs.l.Printf("Repo '%s': commit hash changed, new=%s", name, commitID[:11])
		mbbs.l.Println("Downloading zip archive...")
		buf, err := bb.ZipArchive(ctx, commitID)
		if err != nil {
			return fmt.Errorf("repo '%s': failed to download: %s", name, err)
		}
		// Update repo status
		status.CommitID = commitID
		status.Timestamp = time.Unix(ci.CommitterTimestamp/1000, 0)
		status.Content = buf
		status.Message = ci.Message
		mbbs.l.Printf("Repo '%s' successfully updated", name)
	}
	// Merge all remote statuses into one zip
	z, err := merge(mbbs.statuses)
	if err != nil {
		return fmt.Errorf("merge failed: %v", err)
	}
	mbbs.setZIP(z)
	return nil
}

func (mbbs *MultiBBsync) GetZip() (*types.SaltZip, error) {
	if mbbs.z == nil {
		return nil, errors.New("no zip yet")
	}
	return mbbs.z, nil
}
