package tick

import (
	"context"
	"log"
	"math/rand"
	"time"

	"a.yandex-team.ru/library/go/yandex/unistat"
	"a.yandex-team.ru/library/go/yandex/unistat/aggr"
)

/* Runs sync periodically */
type Periodic struct {
	l         *log.Logger
	name      string
	period    time.Duration
	deltaMax  time.Duration
	s         Syncable
	failCount *unistat.Numeric
}

func NewPeriodic(l *log.Logger, name string, period time.Duration, s Syncable) *Periodic {
	f := unistat.NewNumeric(name+"-sync-fail", 10, aggr.Counter(), unistat.Sum)
	unistat.Register(f)
	return &Periodic{
		deltaMax:  0,
		l:         l,
		name:      name,
		period:    period,
		s:         s,
		failCount: f,
	}
}

func (p *Periodic) WithRandomDelta(max time.Duration) {
	p.deltaMax = max
}

func (p *Periodic) Run(ctx context.Context) error {
	t := time.NewTimer(p.period + randDurationN(p.deltaMax))
	for {
		select {
		case <-ctx.Done():
			p.l.Printf("Periodic sync '%s' stopped: %s", p.name, ctx.Err())
			return ctx.Err()
		case <-t.C:
			p.l.Printf("Synchronizing '%s'...", p.name)
			if err := p.s.Sync(ctx); err != nil {
				p.failCount.Update(1)
				p.l.Printf("Sync '%s' failed: %s", p.name, err)
			}
			nextInterval := p.period + randDurationN(p.deltaMax)
			p.l.Printf("Next sync of '%s' after %s...", p.name, nextInterval.String())
			t.Reset(nextInterval)
		}
	}
}

func randDurationN(d time.Duration) time.Duration {
	n := int(d)
	if n <= 0 {
		return time.Duration(0)
	}
	return time.Duration(rand.Intn(n))
}
