package yasmclient

import (
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"strings"
	"time"
)

const (
	DefaultLocalURL = "http://localhost:11005"
)

type YasmValue struct {
	Name  string      `json:"name"`
	Value interface{} `json:"val"`
}

type YasmMetrics struct {
	Tags   map[string]string `json:"tags"`
	TTL    uint              `json:"ttl"`
	Values []YasmValue       `json:"values"`
}

type YasmResponse struct {
	Status      string `json:"status"`
	ErrorString string `json:"error"`
}

type YasmClient struct {
	ServerURL string
	Client    *http.Client
	Header    map[string]string
}

func NewLocalWithURL(serverURL string) (client *YasmClient) {
	return &YasmClient{
		ServerURL: serverURL,
		Client: &http.Client{
			Timeout: 5 * time.Second,
		},
		Header: make(map[string]string),
	}
}

// NewLocal create client which push events to local yasm service
func NewLocal() (client *YasmClient) {
	return NewLocalWithURL(DefaultLocalURL)
}

func (c *YasmClient) makeRequest(ctx context.Context, json bool, requestBody string) ([]byte, error) {
	req, err := http.NewRequest("POST", c.ServerURL, strings.NewReader(requestBody))
	if err != nil {
		return nil, err
	}
	for k, v := range c.Header {
		req.Header.Set(k, v)
	}
	if json {
		req.Header.Set("Content-Type", "application/json")
	}
	req = req.WithContext(ctx)
	resp, err := c.Client.Do(req)
	if err != nil {
		return nil, err
	}
	defer func() {
		_ = resp.Body.Close()
	}()
	reply, err := ioutil.ReadAll(resp.Body)
	return reply, err
}

func (c *YasmClient) SendMetrics(ctx context.Context, metrics []YasmMetrics) error {
	requestBody, err := json.Marshal(&metrics)
	if err != nil {
		return fmt.Errorf("bad request err:%w", err)
	}
	responseBody, err := c.makeRequest(ctx, true, string(requestBody))
	if err != nil {
		return fmt.Errorf("submit failed err:%w", err)
	}
	var result YasmResponse
	err = json.Unmarshal(responseBody, &result)
	if err != nil {
		return fmt.Errorf("fail to parse request, err:%w", err)
	}
	if result.Status != "ok" {
		return fmt.Errorf("fail to send metrics, staus:%s errmsg: %s", result.Status, result.ErrorString)
	}
	return nil
}
