import {LOCATIONS_URL} from '../config'

const timeout = 30 * 1000; // 30sec

function handleReport(rep) {
    rep.name = rep.name ? rep.name : "-";
    rep.kind = rep.kind ? rep.kind : "-";
    rep.node = rep.node ? rep.node : "-";
    rep.stage = rep.stage ? rep.stage : "-";
    rep.version = rep.version ? rep.version : "-";
    rep.lastTransition = rep.lastTransition ? new Date(rep.lastTransition).toLocaleString() : "-";
    rep.reportTime = rep.reportTime ? new Date(rep.reportTime).toLocaleString() : "-";
}

export class Api {
    constructor(location) {
        this.url = '';
        if (location) {
            this.url = LOCATIONS_URL[location];
            this.location = location
        }
    }

    async getUnit(unitName) {
        const controller = new AbortController();
        const {signal} = controller;
        setTimeout(() => controller.abort(), timeout);
        const unitRes = await fetch(`${this.url}/api/units/${unitName}`, {signal});
        if (unitRes.status !== 200) {
            return {
                err: unitRes.error()
            }
        }
        const unit = await unitRes.json();
        return {
            location: this.location,
            versions: Object.entries(unit.versions).sort((a, b) => b[1] - a[1]),
            stages: Object.entries(unit.stages).sort((a, b) => b[1] - a[1]),
            ready: unit.ready.sort((a, b) => b.count - a.count).map(x => [x.status, x.count]),
            pending: unit.pending.sort((a, b) => b.count - a.count).map(x => [x.status, x.count]),
            name: unitName
        };
    }

    async getKernel() {
        const controller = new AbortController();
        const {signal} = controller;
        setTimeout(() => controller.abort(), timeout);
        const unitRes = await fetch(`${this.url}/api/kernels`, {signal});
        if (unitRes.status !== 200) {
            return {
                err: unitRes.error()
            }
        }
        const unit = await unitRes.json();
        return {
            location: this.location,
            versions: Object.entries(unit.versions).sort((a, b) => b[1] - a[1]),
        };
    }

    async listUnits() {
        const controller = new AbortController();
        const {signal} = controller;
        setTimeout(() => controller.abort(), timeout);
        const unitsResp = await fetch(`${this.url}/api/units`, {signal});
        if (unitsResp.status !== 200) {
            return {
                error: {
                    status: unitsResp.status,
                    message: unitsResp.message
                }
            };
        }
        const units = await unitsResp.json();
        const unitsMap = {};
        for (const kind of Object.keys(units.units)) {
            unitsMap[kind] = units.units[kind].units;
        }
        return {
            units: unitsMap,
            location: this.location,
        };
    }

    async listHosts(req) {
        const controller = new AbortController();
        const {signal} = controller;
        setTimeout(() => controller.abort(), timeout);
        const unitsResp = await fetch(`${this.url}/api/hosts/`, {
            body: JSON.stringify({
                hostInfo: {
                    hostname: req.hostname,
                    num: req.num,
                    walleProject: req.walleProject,
                    walleTags: req.walleTag ? [req.walleTag] : [],
                    netSwitch: req.netSwitch,
                    gencfgGroups: req.gencfgGroup ? [req.gencfgGroup] : [],
                    location: req.location,
                    dc: req.dc,
                    kernelRelease: req.kernel
                }
            }),
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            signal
        });
        if (unitsResp.status !== 200) {
            return {
                error: {
                    status: unitsResp.status,
                    message: unitsResp.message
                }
            };
        }
        return (await unitsResp.json()).hosts || [];
    }

    async getHost(hostname) {
        const controller = new AbortController();
        const {signal} = controller;
        setTimeout(() => controller.abort(), timeout);
        const unitsResp = await fetch(`/api/hosts/${hostname}`, {signal});
        if (unitsResp.status !== 200) {
            return {
                error: {
                    status: unitsResp.status,
                    message: unitsResp.message
                }
            };
        }
        return await unitsResp.json();
    }

    async getReports(node, unit, stage, version, limit, offset, ready, pending) {
        const controller = new AbortController();
        const {signal} = controller;
        setTimeout(() => controller.abort(), timeout);
        const res = await fetch(`${this.url}/api/reports`, {
            body: JSON.stringify({node, unit, stage, version, limit, offset, ready, pending}),
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            signal
        });
        const json = await res.json();
        if (!json.reports) {
            return {reports: []}
        }
        for (let i = 0; i < json.reports.length; i++) {
            handleReport(json.reports[i]);
        }
        return {reports: json.reports};
    }
}
